/*
This software is subject to the license described in the license.txt file included with this software distribution. You may not use this file except in compliance with this license.
Copyright  Dynastream Innovations Inc. 2017
All rights reserved.
*/

// Control how FE data from page 16 is handled and stored (General FE Data)
// Protocol defined in the fitness equiptment documentation on www.thisisant.com
// Parse the message payload and set the values in the data model
class AntPage16 extends FitnessEquipmentMainPage
{

    const PAGE_NUM = 16;
    const EQUIPMENT_TYPE_BIT_FIELD = 0x1F;
    const HR_DATA_SOURCE_BIT_FIELD = 0x03;
    const DISTANCE_TRAVELED_ENABLED_BIT_FIELD = 0x04;
    const VIRTUAL_SPEED_BIT_FIELD = 0x08;
    // Invalid values
    const INVALID_SPEED = 0xFFF;
    const INVALID_HEART_RATE = 0xFF;
    const INVALID_HR_DATA_SOURCE = 0;

    hidden var _equipmentTypes = { 16=>"General", 19=>"Treadmill", 20=>"Elliptical", 21=>"Stationary Bike", 22=>"Rower", 23=>"Climber", 24=>"Nordic Skier", 25=>"Trainer" };
    hidden var _equipmentType;
    hidden var _elapsedTime;
    hidden var _distanceTraveled;
    hidden var _speed;
    hidden var _heartRate;
    hidden var _heartRateDataSource;
    hidden var _distanceTraveledEnabled;
    hidden var _virtualSpeedFlag;

    function initialize(payload){

        FitnessEquipmentMainPage.initialize(payload);
        self._equipmentType = _equipmentTypes[16];
        self._elapsedTime = "--";
        self._distanceTraveled = "--";
        self._speed = "--";
        self._heartRate = "--";
        self._heartRateDataSource = "--";
        self._distanceTraveledEnabled = "--";
        self._virtualSpeedFlag = "--";
    }


    function parse(){

        FitnessEquipmentMainPage.parse();
        setStateData();
        setEquipmentType();
        setElapsedTime();
        setDistanceTraveled();
        setSpeed();
        setHeartRate();
        setHrDataSource();
        setDistanceTraveledEnabled();
        setVirtualSpeedFlag();
    }


    function updateAppData(){

        FitnessEquipmentMainPage.updateAppData();
        appData.setEquipmentType( self._equipmentType );
        appData.setElapsedTime( self._elapsedTime );
        appData.setDistanceTraveled( self._distanceTraveled );
        appData.setSpeed( self._speed );
        appData.setHeartRate( self._heartRate );
        appData.setHrDataSource( self._heartRateDataSource );
        appData.setDistanceTraveledEnabled( self._distanceTraveledEnabled );
        appData.setVirtualSpeedFlag( self._virtualSpeedFlag );
    }


    function setEquipmentType(){

        var type = ( self._currentPayload[1] & EQUIPMENT_TYPE_BIT_FIELD );
        self._equipmentType = _equipmentTypes[type.toNumber()];
    }


    function setElapsedTime(){

        var quarterSeconds = self._currentPayload[2].toNumber();
        self._elapsedTime = quarterSeconds * 4;
    }


    function setDistanceTraveled(){

        self._distanceTraveled = self._currentPayload[3].toNumber();
    }


    function setSpeed(){

        var millimetersPerSecond = self._currentPayload[4].toNumber() + ( self._currentPayload[5].toNumber() * 256 );
        var incommingSpeed = millimetersPerSecond * 1000;
        if( incommingSpeed == INVALID_SPEED ){ return; }
        self._speed = incommingSpeed;
    }


    function setHeartRate(){

        var incommingHrDataSource = self._currentPayload[7].toNumber() & HR_DATA_SOURCE_BIT_FIELD;
        var incommingHeartRate = self._currentPayload[6];
        if( invalidHeartRate( incommingHeartRate, incommingHrDataSource )){ return; }
        self._heartRate = incommingHeartRate;
    }


    function invalidHeartRate( heartRateValue, heartRateSource ){

        return heartRateValue == INVALID_HEART_RATE || heartRateSource == INVALID_HR_DATA_SOURCE;
    }


    function setHrDataSource(){

        var sourceVal = ( self._currentPayload[7].toNumber() & HR_DATA_SOURCE_BIT_FIELD );
        self._heartRateDataSource = sourceVal;
    }


    function setDistanceTraveledEnabled(){

        self._distanceTraveledEnabled = ( self._currentPayload[7].toNumber() & DISTANCE_TRAVELED_ENABLED_BIT_FIELD ) >> 2;
    }


    function setVirtualSpeedFlag(){

        self._virtualSpeedFlag = ( self._currentPayload[7].toNumber() & VIRTUAL_SPEED_BIT_FIELD ) >> 3;
    }

}