/*
This software is subject to the license described in the license.txt file included with this software distribution. You may not use this file except in compliance with this license.
Copyright  Dynastream Innovations Inc. 2017
All rights reserved.
*/

// A class to represent all of the applicable data for the application
// There should only ever be one instance of this class created
// The one instance is available across the whole app
static class AppData
{
    // Specific Ant Channel
    var gFit;
    var scanner;

    //Ensure only one instance can be created
    hidden static var _hasBeenInstantiated = false;

    // Device Search data
    hidden const MAX_DEVICES = 75;
    hidden var _devices = {};
    hidden var _searchState = true;

    // General page data
    hidden var _equipmentState;

    // Data from page 16
    hidden const ELAPSED_TIME_RANGE = 64;
    hidden const DISTANCE_TRAVELED_RANGE = 256;
    hidden var _equipmentType;
    hidden var _elapsedTime;
    hidden var _distanceTraveled;
    hidden var _speed;
    hidden var _heartRate;
    hidden var _hrDataSource;
    hidden var _distanceTraveledEnabled;
    hidden var _virtualSpeedFlag;

    // Data from page 25 and 21
    // Common between 21 & 25:
    hidden var _instantaneousPower;
    hidden var _instantaneousCadence;
    // 21 specific:
    // 25 specific:
    hidden const ACCUMULATED_POWER_RANGE = 65536;
    hidden var _updateEventCount;
    hidden var _accumulatedPower;
    hidden var _bicyclePowerCalibration;
    hidden var _resistanceCalibration;
    hidden var _userConfiguration;
    hidden var _targetPowerLimits;

    // FIT data
    hidden var _session;
    hidden var _powerFitField;
    hidden var _heartRateFitField;
    hidden var _cadenceFitField;

    function initialize(){

        if( _hasBeenInstantiated ){
            Toybox.System.println("Only one AppData instace should ever be created.");
            throw new Toybox.Lang.Exception.InitializerError();
        }

        _hasBeenInstantiated = true;

        // Set defaults
        setDefaultData();
        createSession();
    }


    // FIT file stuff
    function createSession(){

    self._session = Toybox.ActivityRecording.createSession({
            :sport => Toybox.ActivityRecording.SPORT_FITNESS_EQUIPMENT,
            :subSport => Toybox.ActivityRecording.SUB_SPORT_INDOOR_CYCLING,
            :name=>"Fitness Equipment"
            });
        self._powerFitField = self._session.createField("Instantaneous Power", 1, Toybox.FitContributor.DATA_TYPE_UINT16, {
            :units => "Watts",
            :mesgType => Toybox.FitContributor.MESG_TYPE_RECORD
            });
        self._heartRateFitField = self._session.createField("Heart Rate", 2, Toybox.FitContributor.DATA_TYPE_UINT8, {
            :units => "BPM",
            :mesgType => Toybox.FitContributor.MESG_TYPE_RECORD
            });
        self._cadenceFitField = self._session.createField("Instantaneous Cadence", 3, Toybox.FitContributor.DATA_TYPE_UINT16, {
            :units => "RPM",
            :mesgType => Toybox.FitContributor.MESG_TYPE_RECORD
            });
    }


    function discardSession(){

        self._session.discard();
        self._session = null;
        createSession();
    }


    function sessionIsRecording(){

        return self._session.isRecording();
    }


    function saveSession(){

        self._session.save();
        self._session = null;
        createSession();
    }


    function startSession(){

        return self._session.start();
    }


    function stopSession(){

        return self._session.stop();
    }


    function updateFitFile(){

        if( sessionIsRecording() ){

            if( self.getInstantaneousPower().toString().hashCode() != "--".hashCode() ){
                self._powerFitField.setData( self.getInstantaneousPower() );
            }
            if( self.getHeartRate().toString().hashCode() != "--".hashCode() ){
                self._heartRateFitField.setData( self.getHeartRate() );
            }
            if( self.getInstantaneousCadence().toString().hashCode() != "--".hashCode() ){
                self._cadenceFitField.setData( self.getInstantaneousCadence() );
            }

        }

    }


    // Ant channel management
    function connectToDevice( deviceNumber ){

        self.gFit = new GFitModule( deviceNumber );
        self.gFit.open();
    }


    function openScanner(){

        self.scanner = new Scanner();
        self.scanner.open();
    }


    function releaseScanner(){

        self.scanner.release();
    }


    // Device search data
    function setSearchState( state ){

        self._searchState = state;
    }

    function getSearchState(){

        return self._searchState;
    }

    function getDevices(){

        return self._devices;
    }


    function setDeviceFromMessage( message ){

        if( shouldIgnoreDevice( message ) ){ return; }
        self._devices.put(message.deviceNumber, message.rssi);
    }


    function shouldIgnoreDevice( message ){

        return self._devices.size() >= MAX_DEVICES && !self._devices.hasKey(message.deviceNumber);
    }


    function getDeviceInfo( deviceNumber ){

        return self._devices[deviceNumber];
    }


    function removeDevice( deviceNumber ){

        self._devices.remove( deviceNumber );
    }


    function removeAllDevices(){

        self._devices = {};
    }


    // General page data
    function setDefaultData(){

        self._elapsedTime = "--";
        self._distanceTraveled = "--";
        self._speed = "--";
        self._heartRate = "--";
        self._instantaneousPower = "--";
        self._instantaneousCadence = "--";
        self._accumulatedPower = "--";
    }


    function getEquipmentState(){

        return self._equipmentState;
    }


    function setEquipmentState( equipmentState ){

        self._equipmentState = equipmentState;
    }


    // Data from page 16
    function getEquipmentType(){

        return self._equipmentType;
    }


    function setEquipmentType( equipmentType ){

        self._equipmentType = equipmentType;
    }


    function getElapsedTime(){

        return self._elapsedTime;
    }


    function setElapsedTime( elapsedTime ){

        self._elapsedTime = elapsedTime;
    }


    function getDistanceTraveled(){

        return self._distanceTraveled;
    }


    function setDistanceTraveled( distanceTraveled ){

        self._distanceTraveled = distanceTraveled;
    }


    function getSpeed(){

        return self._speed;
    }


    function setSpeed( speed ){

        self._speed = speed;
    }


    function getHeartRate(){

        return self._heartRate;
    }


    function setHeartRate( heartRate ){

        self._heartRate = heartRate;
    }


    function getHrDataSource(){

        return self._hrDataSource;
    }


    function setHrDataSource( hrDataSource ){

        self._hrDataSource = hrDataSource;
    }


    function getDistanceTraveledEnabled(){

        return self._distanceTraveledEnabled;
    }


    function setDistanceTraveledEnabled( distanceTraveledEnabled ){

        self._distanceTraveledEnabled = distanceTraveledEnabled;
    }


    function getVirtualSpeedFlag(){

        return self._virtualSpeedFlag;
    }


    function setVirtualSpeedFlag( virtualSpeedFlag ){

        self._virtualSpeedFlag = virtualSpeedFlag;
    }


    // Data from page 21 and 25
    function getInstantaneousCadence(){

        return self._instantaneousCadence;
    }


    function setInstantaneousCadence( instantaneousCadence ){

        self._instantaneousCadence =  instantaneousCadence;
    }


    function getInstantaneousPower(){

        return self._instantaneousPower;
    }


    function setInstantaneousPower( instantaneousPower ){

        self._instantaneousPower = instantaneousPower;
    }


    // Data specific to page 25
    function getUpdateEventCount(){

        return self._updateEventCount;
    }


    function setUpdateEventCount( updateEventCount ){

        self._updateEventCount = updateEventCount;
    }


    function getAccumulatedPower(){

        return self._accumulatedPower;
    }


    function setAccumulatedPower( accumulatedPower ){

        self._accumulatedPower = accumulatedPower;
    }


    function getBicyclePowerCalibration(){

        return self._bicyclePowerCalibration;
    }


    function setBicyclePowerCalibration( bicyclePowerCalibration ){

        self._bicyclePowerCalibration = bicyclePowerCalibration;
    }


    function getResistanceCalibration(){

        return self._resistanceCalibration;
    }

    function setResistanceCalibration( resistanceCalibration ){

        self._resistanceCalibration = resistanceCalibration;
    }


    function getUserConfiguration(){

        return self._userConfiguration;
    }


    function setUserConfiguration( userConfiguration ){

        self._userConfiguration = userConfiguration;
    }


    function getTargetPowerLimits(){

        return self._targetPowerLimits;
    }


    function setTargetPowerLimits( targetPowerLimits ){

        self._targetPowerLimits = targetPowerLimits;
    }

}