//
// Copyright  Dynastream Innovations Inc. 2017
// Subject to Garmin SDK License Agreement and Wearables
// Application Developer Agreement.
//
/*
This software is subject to the license described in the license.txt file included with this software distribution. You may not use this file except in compliance with this license.
Copyright  Dynastream Innovations Inc. 2017
All rights reserved.
*/

// Menu screen to display gfit modules and indicate the strongest signals
class DevicesView extends Toybox.WatchUi.View
{

    var devicesController;
    const Y_COMPONENT = 1;
    hidden var _refreshTimer = new Toybox.Timer.Timer();
    hidden var _selectedDeviceIndex;
    hidden var _devicesList = {};
    hidden var _menuRows;
    hidden var _screenHeight;
    hidden var _screenWidth;
    hidden var _fontDevices;
    hidden var _fontTitle;
    hidden var _numberOfRows;
    hidden var _rowHeight;
    hidden var _rowOffsetFromTop;
    hidden var _scrollFromTop;
    hidden var _cursorFromTop;
    hidden var _closeProximityCutoff;

    enum{
            X,
            Y
    }

    function initialize(){

        View.initialize();
        self.devicesController = new DevicesController();
        self._selectedDeviceIndex = 0;
        self._scrollFromTop = 0;
        self._cursorFromTop = 0;
        self._menuRows = {};
        self._fontDevices = Toybox.Graphics.FONT_MEDIUM;
        self._fontTitle = Toybox.Graphics.FONT_LARGE;
        devicesController.openScan();
    }


    function onLayout(dc) {

        setLayout(Rez.Layouts.DevicesLayout(dc));
        // Get values for formatting
        self._screenHeight = dc.getHeight();
        self._screenWidth = dc.getWidth();
        self._rowOffsetFromTop = dc.getTextDimensions("Devices", self._fontTitle)[Y_COMPONENT];
        self._rowHeight = dc.getTextDimensions("123ABC", self._fontDevices)[Y_COMPONENT];
        self._numberOfRows = Math.floor(((self._screenHeight - self._rowOffsetFromTop) / self._rowHeight));
        drawScrollSymbol( dc );
    }


    function onShow() {

        if( appData.getSearchState() == true ){
            cleanMenu();
            devicesController.startSearchTimer( 30000 );
            self._refreshTimer.start( method( :refreshDevices ), 1500, true );
            refreshDevices();
        }
    }


    function onHide(){

        devicesController.stopTimer();
        devicesController.removeAllDevices();
        devicesController.resetScan();
        self._refreshTimer.stop();
    }


    // Update the view
    function onUpdate(dc) {

        View.onUpdate(dc);
        drawScrollSymbol( dc );
        // Track all of devices visible to the watch
        var deviceNumbers = self._devicesList.keys();

        for(var rowNumber = 0; rowNumber < deviceNumbers.size(); rowNumber++){
            var deviceNumber = deviceNumbers[rowNumber];
            var row;
            // Update rows for devices already in the menu
            if( self._menuRows.hasKey( deviceNumber )){
                row = self._menuRows.get( deviceNumber );
                row.setLocation( self._screenWidth/3, self._rowOffsetFromTop + (rowNumber * self._rowHeight) + (self._scrollFromTop * self._rowHeight));
                row.setText( deviceNumber.toString());
            }
            // Add new devices to the menu
            else{
                row = new DeviceRow( {
                    :height => self._rowHeight,
                    :identifier => deviceNumber,
                    :locX => self._screenWidth/3,
                    :locY => self._rowOffsetFromTop + (rowNumber * self._rowHeight) + (self._scrollFromTop * self._rowHeight),
                    :text => deviceNumber.toString(),
                    :font => self._fontDevices,
                    :textColor => Toybox.Graphics.COLOR_WHITE,
                    :selectedColor => Toybox.Graphics.COLOR_BLUE
                } );

                self._menuRows.put(row.identifier, row);
            }
            row.addRssiSample( self._devicesList.get(deviceNumber) );
        }

        //Calculate a the strongest signal and update the menu to depict it
        if( appData.getSearchState() == true ){
            self._closeProximityCutoff = calculateProximityCutoff();
        }
        for(var rowNumber = 0; rowNumber < deviceNumbers.size(); rowNumber++){
            var deviceNumber = deviceNumbers[rowNumber];
            var row = self._menuRows.get( deviceNumber );
            updateRowStatus( row, deviceNumbers );
            row.drawRow( dc );
        }
    }


    // Add or remove the selection indicator
    function updateRowStatus( row, devices ){

        if( row.rssiAverage() >= self._closeProximityCutoff ){
            row.addIndicator();
        }
        else{ row.removeIndicator(); }

        if( rowSelectedInMenu( row, devices )){ row.addSelection(); }
        else{ row.removeSelection(); }

        if( rowIsOffScreen( row ) ){ row.hide(); }
        else{ row.show(); }
    }

    // Determine of a row is selected or not
    function rowSelectedInMenu( row, devices ){ return row.identifier == devices[self._selectedDeviceIndex];  }

    // Determine if a row should be hidden
    function rowIsOffScreen( row ){ return row.locY < self._rowOffsetFromTop; }


    // Calculate a cutoff to define a "close proximity" device
    function calculateProximityCutoff(){
        var rows = self._menuRows.values();
        var largest = 0;
        for( var i = 0; i < rows.size(); i++){
            if(rows[i].rssiAverage() > largest ){
                largest = rows[i].rssiAverage();
            }
        }
        return largest;
    }


    // Add new devices to the menu if they have been discovered since the last update
    function refreshDevices(){

        var devices = devicesController.getDevices();
        var deviceNumbersToAdd = devicesController.getDevices().keys();
        for(var i = 0; i < deviceNumbersToAdd.size(); i++){
            self._devicesList.put( deviceNumbersToAdd[i], devices.get( deviceNumbersToAdd[i] ) );
        }
        Toybox.WatchUi.requestUpdate();
    }


    // Called when the user swipes down or presses the page down button
    function scrollDown(){

        if( self._devicesList.size() > 0)
        {
            self._selectedDeviceIndex++;
            if( self._selectedDeviceIndex > ( self._menuRows.size() - 1 )){
                self._selectedDeviceIndex = ( self._menuRows.size() - 1 );
            }
            else if( self._cursorFromTop == (self._numberOfRows - 1) ){
                movePageUp();
            }
            else{
                self._cursorFromTop++;
            }

            Toybox.WatchUi.requestUpdate();
        }
    }

    // Called when the user swipes up or presses the page up button
    function scrollUp(){

        if( self._devicesList.size() > 0){
            self._selectedDeviceIndex--;

            if( self._selectedDeviceIndex < 0 ){
                self._selectedDeviceIndex = 0;
            }
            else if( self._cursorFromTop == 0 && self._scrollFromTop < 0 ){
                movePageDown();
            } else{
                self._cursorFromTop--;
            }

            Toybox.WatchUi.requestUpdate();
       }
    }


    // Adjust the rows that are visable on screen, shift all rows up one spot
    function movePageUp(){

        var devices = self._devicesList.keys();
        self._scrollFromTop--;
        for(var rowNumber = 0; rowNumber < devices.size(); rowNumber++){
            var deviceNumber = devices[rowNumber];
            var row = self._menuRows.get(deviceNumber);
            row.setLocation(self._screenWidth/3, row.locY + (self._scrollFromTop * self._rowHeight) );
        }
    }


    // Adjust the rows that are visable on screen, shift all rows down one spot
    function movePageDown(){

        var devices = self._devicesList.keys();
        self._scrollFromTop++;
        for(var rowNumber = 0; rowNumber < devices.size(); rowNumber++){
            var deviceNumber = devices[rowNumber];
            var row = self._menuRows.get(deviceNumber);
            row.setLocation(self._screenWidth/3, row.locY + (self._scrollFromTop * self._rowHeight) );
        }
    }


    // Select a device to pair to
    function selectDevice(){

        if( self._devicesList.size() > 0){
            var deviceNumber = self._devicesList.keys()[self._selectedDeviceIndex];
            devicesController.openChannel( deviceNumber );
        }
    }


    function screenHeight(){
        return self._screenHeight;
    }


    function screenWidth(){
        return self._screenWidth;
    }


    // Remove devices from the menu if they have not been heard from in a while
    function cleanMenu(){

        self._menuRows = {};
        self._devicesList = {};
        self._selectedDeviceIndex = 0;
        self._scrollFromTop = 0;
        self._cursorFromTop = 0;
    }

    function drawScrollSymbol( dc ){

        dc.setColor(Toybox.Graphics.COLOR_DK_GRAY, Toybox.Graphics.COLOR_BLACK);
        var x = self._screenWidth*7/8;
        var y = self._screenHeight/2;
        var width = self._screenWidth/15;
        var height = self._screenHeight/2;
        var p1 = [x, y-height/2];
        var p1a = [x+width/2, y-height/2+height/5];
        var p1b = [x-width/2, y-height/2+height/5];
        var p2 = [x, y+height/2];
        var p2a = [x+width/2, y+height/2-height/5];
        var p2b = [x-width/2, y+height/2-height/5];
        dc.drawLine(p1[X], p1[Y], p1a[X], p1a[Y]);
        dc.drawLine(p1[X], p1[Y], p1b[X], p1b[Y]);
        dc.drawLine(p2[X], p2[Y], p2a[X], p2a[Y]);
        dc.drawLine(p2[X], p2[Y], p2b[X], p2b[Y]);
        dc.drawLine(p1[X], p1[Y], p2[X], p2[Y]);
    }

}