/*
This software is subject to the license described in the license.txt file
included with this software distribution. You may not use this file except in compliance
with this license.

Copyright (c) Dynastream Innovations Inc. 2017
All rights reserved.
*/

#include <stdbool.h>
#include <stdint.h>
#include <string.h>

#include "gfit_defines.h"
#include "gfit_interface.h"

#include "app_error.h"
#include "ant_stack_config.h"
#include "ant_parameters.h"
#include "app_scheduler.h"
#include "softdevice_handler.h"
#include "softdevice_handler_appsh.h"
#include "ble.h"
#include "app_timer.h"
#include "bsp.h"
#include "boards.h"
#include "nrf_nvic.h"
#include "nrf_sdm.h"
#include "nrf_soc.h"
#include "nrf_log.h"
#include "nrf_log_ctrl.h"
#include "sdk_config.h"

// Scheduler settings
#define SCHED_MAX_EVENT_DATA_SIZE       MAX(ANT_STACK_EVT_MSG_BUF_SIZE, BLE_STACK_EVT_MSG_BUF_SIZE)
#define SCHED_QUEUE_SIZE                50

#define BLE_CONN_EVT_LENGTH             15                              /**< 15x1.25ms = 18.75ms (Default = 3*1.25 = 3.75ms) */
#define IS_SRVC_CHANGED_CHARACT_PRESENT 0                               /**< Include or not the service_changed characteristic. if not enabled, the server's database cannot be changed for the lifetime of the device*/

// Proximity pairing settings
// For desk testing with the development kit, the default values for this demo are set to very tight thresholds,
// as this allows experimenting with modules on the same desk (e.g. simulated HRMs).
#define DEMO_SEARCH_THRESHOLD           -31                             /**< Searching threshold. -31 dBm .*/
#define DEMO_PAIR_THRESHOLD             -51                             /**< Pairing threshold. -51dBm .*/

#define DEMO_MFG_SPECIFIC_PAGE          ((uint8_t) 0xE3)                /**< Manufacturer specific page .*/

// FE simulation
#define SIM_BIKE_SPEED                  ((uint16_t) 4)                  /**< Simulated fixed speed: 4 m/s ~ 14.4 km/h */
#define SIM_BIKE_CADENCE                ((uint8_t) 90)                  /**< Simulated fixed cadence: 90 rpm. */
#define SIM_BIKE_POWER                  ((uint16_t) 120)                /**< Simulated fixed power: 120 w. */
#define TIMER_TICKS                     APP_TIMER_TICKS(1000)           /**< Timer interval for simulated FE events (1000ms) */
APP_TIMER_DEF(m_timer_id);
gfit_state_t m_gfit_state;


gfit_bike_data_update_t m_bike_trainer_data;                            /**< Stationary Bike Data */

uint8_t m_counter;                      /**< Manufacturer specific data. */
uint8_t m_payload[ANT_STANDARD_DATA_PAYLOAD_SIZE] = {DEMO_MFG_SPECIFIC_PAGE, 0, 0, 0, 0, 0, 0, 0};

/**@brief Function for dispatching a system event to interested modules.
*
* @details This function is called from the System event interrupt handler after a system
*          event has been received.
*
* @param[in]   sys_evt   System stack event.
*/
static void sys_evt_dispatch(uint32_t sys_evt)
{
    gfit_lib_return_t ret = gfit_sys_event_handler(sys_evt);
    APP_ERROR_CHECK_BOOL(GFIT_SUCCESS == ret.gfit_response_code);
}

/**@brief Handle and display heart rate events
 * @param[in]   rx_packet   Heart rate packet from G.FIT
 */
static void display_heart_rate(gfit_hrp_evt_ret_t* rx_packet)
{
    if ( hr_scan_rx == rx_packet->header )
    {
        // Packet contains information about heart rate straps visible on scan.
        // This demo uses RSSI based pairing, so the scan packets are just for debug purposes,
        // to confirm that HRM straps around G.FIT are being received
        if( protocol_ant == rx_packet->evt.hr_scan_evt.protocol )
        {
            NRF_LOG_DEBUG("Rx Scan Packet (ANT+ %02x-%02x) RSSI: %d\r\n",
                    rx_packet->evt.hr_scan_evt.hrm_id[0],
                    rx_packet->evt.hr_scan_evt.hrm_id[1],
                    rx_packet->evt.hr_scan_evt.rssi);
        }
        else if( protocol_ble == rx_packet->evt.hr_scan_evt.protocol )
        {
            NRF_LOG_DEBUG("Rx Scan Packet (BLE %02x-%02x)  RSSI: %d\r\n",
                    rx_packet->evt.hr_scan_evt.hrm_id[0],
                    rx_packet->evt.hr_scan_evt.hrm_id[1],
                    rx_packet->evt.hr_scan_evt.rssi);
        }
        // Optionally print full 6-byte Device ID at DEBUG level
        NRF_LOG_DEBUG("Device ID:\r\n");
        NRF_LOG_HEXDUMP_DEBUG(rx_packet->evt.hr_scan_evt.hrm_id, GFIT_HRM_ID_LENGTH);
    }
    else if ( hr_paired_rx == rx_packet->header )
    {
        // Packet contains information about paired heart rate strap
        // Display HR value on console
        NRF_LOG_INFO("Rx Paired HR Packet (%02x-%02x  RSSI %ddBm) HR: %d\r\n",
                rx_packet->evt.hr_paired_evt.hrm_id[0],
                rx_packet->evt.hr_paired_evt.hrm_id[1],
                rx_packet->evt.hr_paired_evt.rssi,
                rx_packet->evt.hr_paired_evt.heart_rate);
        // Optionally print full 6-byte Device ID at DEBUG level
        NRF_LOG_DEBUG("Device ID:\r\n");
        NRF_LOG_HEXDUMP_DEBUG(rx_packet->evt.hr_paired_evt.hrm_id, GFIT_HRM_ID_LENGTH);
    }
    else
    {
        // Not an HR event. Do nothing.
    }
}

/**@brief Function for processing ANT channel events.
 * @param[in] p_ant_evt Pointer to ANT event
 */
static void ant_event_handler(ant_evt_t* p_ant_evt)
{
    gfit_lib_return_t ret;
    gfit_hrp_evt_ret_t rx_packet;

    // Pass all events managed by G.FIT, as well as device level events (channel field may be zero in these)
    // to the G.FIT ANT event handler
    if ( ( 0 == p_ant_evt->channel) || ( GFIT_ANT_CHANNEL_RESERVED_MIN <= p_ant_evt->channel ) )
    {
        ret = gfit_ant_event_handler(p_ant_evt, &rx_packet);
        APP_ERROR_CHECK_BOOL(GFIT_SUCCESS == ret.gfit_response_code);

        display_heart_rate(&rx_packet);
    }

    // Handle messages received from ANT+ FE-C display
    if ( ( GFIT_FEC_ANT_CHANNEL_NUMBER_PRIMARY == p_ant_evt->channel ) ||
         ( GFIT_FEC_ANT_CHANNEL_NUMBER_DIVERSIFICATION == p_ant_evt-> channel ) )
    {
        if( EVENT_RX == p_ant_evt->event )
        {
            NRF_LOG_INFO("Received message from ANT+ FE-C display.  Payload:\r\n");
            ANT_MESSAGE* p_msg = (ANT_MESSAGE*) p_ant_evt->msg.evt_buffer;
            NRF_LOG_HEXDUMP_INFO(p_msg->ANT_MESSAGE_aucPayload, ANT_STANDARD_DATA_PAYLOAD_SIZE);
            // The application can do additional processing of the payload here, e.g.,
            // decoding ANT+ FE-C control pages
        }
    }
}

/**@brief Function for processing BLE events.
 * @param[in] p_ble_evt Pointer to BLE event
 */
static void ble_event_handler(ble_evt_t *p_ble_evt)
{
   gfit_lib_return_t ret;
   gfit_hrp_evt_ret_t rx_packet;

   // Pass all BLE events to the G.FIT BLE event handler
   ret = gfit_ble_event_handler(p_ble_evt, &rx_packet);
   APP_ERROR_CHECK_BOOL(GFIT_SUCCESS == ret.gfit_response_code);

   display_heart_rate(&rx_packet);
}


/**@brief Custom GFIT event handler
 *
 */
void gfit_custom_event_handler(uint8_t event)
{
    NRF_LOG_INFO("Received custom event over ANT+ FE-C channel: %d\r\n", event);
}


/**@brief Function for ANT (and BLE) stack initialization.
*
* @details Initializes the SoftDevice and the ANT event interrupt.
*/
static void softdevice_init(void)
{
   uint32_t err_code;

   nrf_clock_lf_cfg_t clock_lf_cfg = NRF_CLOCK_LFCLKSRC;

   // Initialize SoftDevice
   SOFTDEVICE_HANDLER_APPSH_INIT(&clock_lf_cfg, softdevice_evt_schedule);

   // Fetch the start address of the application RAM.
   uint32_t ram_start = 0;
   err_code = softdevice_app_ram_start_get(&ram_start);
   APP_ERROR_CHECK(err_code);

   // BLE stack configuration
   ble_cfg_t ble_cfg;

   // Configure the number of custom UUIDS. - Custom UUID Required for Buttonless DFU Service
   memset(&ble_cfg, 0, sizeof(ble_cfg));
   ble_cfg.common_cfg.vs_uuid_cfg.vs_uuid_count = 1;
   err_code = sd_ble_cfg_set(BLE_COMMON_CFG_VS_UUID, &ble_cfg, ram_start);
   APP_ERROR_CHECK(err_code);

   // Configure the maximum number of connections.
   memset(&ble_cfg, 0, sizeof(ble_cfg));
   ble_cfg.gap_cfg.role_count_cfg.periph_role_count  = GFIT_PERIPHERAL_LINK_COUNT;
   ble_cfg.gap_cfg.role_count_cfg.central_role_count = GFIT_CENTRAL_LINK_COUNT;
   ble_cfg.gap_cfg.role_count_cfg.central_sec_count  = GFIT_CENTRAL_LINK_COUNT;
   err_code = sd_ble_cfg_set(BLE_GAP_CFG_ROLE_COUNT, &ble_cfg, ram_start);
   APP_ERROR_CHECK(err_code);

   // Configure the maximum ATT MTU.
   memset(&ble_cfg, 0x00, sizeof(ble_cfg));
   ble_cfg.conn_cfg.conn_cfg_tag                 = GFIT_CONN_CFG_TAG_CENTRAL;
   ble_cfg.conn_cfg.params.gatt_conn_cfg.att_mtu = NRF_BLE_GATT_MAX_MTU_SIZE;
   err_code = sd_ble_cfg_set(BLE_CONN_CFG_GATT, &ble_cfg, ram_start);
   APP_ERROR_CHECK(err_code);

   // Configure the maximum event length.
   memset(&ble_cfg, 0x00, sizeof(ble_cfg));
   ble_cfg.conn_cfg.conn_cfg_tag                     = GFIT_CONN_CFG_TAG_CENTRAL;
   ble_cfg.conn_cfg.params.gap_conn_cfg.event_length = BLE_CONN_EVT_LENGTH;
   ble_cfg.conn_cfg.params.gap_conn_cfg.conn_count   = GFIT_CENTRAL_LINK_COUNT;
   err_code = sd_ble_cfg_set(BLE_CONN_CFG_GAP, &ble_cfg, ram_start);
   APP_ERROR_CHECK(err_code);

   // Enable BLE stack.
   err_code = softdevice_enable(&ram_start);
   APP_ERROR_CHECK(err_code);

   // Subscribe for BLE events.
   err_code = softdevice_ble_evt_handler_set(ble_event_handler);
   APP_ERROR_CHECK(err_code);

   // Subscribe for ANT events.
   err_code = softdevice_ant_evt_handler_set(ant_event_handler);
   APP_ERROR_CHECK(err_code);

   // Register with the SoftDevice handler module for BLE events.
   err_code = softdevice_sys_evt_handler_set(sys_evt_dispatch);
   APP_ERROR_CHECK(err_code);

   // Configure ANT stack.
   err_code = ant_stack_static_config();
   APP_ERROR_CHECK(err_code);
}

static void fe_fields_set(void)
{
    gfit_lib_return_t ret;

    // Send all updated bike FE metrics to the G.FIT
    ret = gfit_fep_set_bike_data(m_bike_trainer_data);
    APP_ERROR_CHECK_BOOL( GFIT_SUCCESS == ret.gfit_response_code );

    // Once data has been set to G.FIT, reset all update fields
    m_bike_trainer_data.elapsed_time_seconds_updated = false;
    m_bike_trainer_data.total_distance_meters_updated = false;
    m_bike_trainer_data.instantaneous_speed_millimeters_per_second_updated = false;
    m_bike_trainer_data.instantaneous_cadence_half_revolutions_per_min_updated = false;
    m_bike_trainer_data.instantaneous_power_watts_updated = false;
    m_bike_trainer_data.accumulated_power_watts_updated = false;
}

static void fe_fields_init(void)
{
    // Configure which FE fields will be supported by this piece of fitness equipment before startup
    m_bike_trainer_data.elapsed_time_seconds_updated = true;
    m_bike_trainer_data.total_distance_meters_updated = true;
    m_bike_trainer_data.instantaneous_speed_millimeters_per_second_updated = true;
    m_bike_trainer_data.instantaneous_cadence_half_revolutions_per_min_updated = true;
    m_bike_trainer_data.instantaneous_power_watts_updated = true;
    m_bike_trainer_data.accumulated_power_watts_updated = true;
}

static void fe_fields_reset(void)
{
    // Reset values
    m_bike_trainer_data.elapsed_time_seconds = 0;
    m_bike_trainer_data.elapsed_time_seconds_updated = true;

    m_bike_trainer_data.total_distance_meters = 0;
    m_bike_trainer_data.total_distance_meters_updated = true;

    m_bike_trainer_data.instantaneous_speed_millimeters_per_second = 0;
    m_bike_trainer_data.instantaneous_speed_millimeters_per_second_updated = true;

    m_bike_trainer_data.instantaneous_cadence_half_revolutions_per_min = 0;
    m_bike_trainer_data.instantaneous_cadence_half_revolutions_per_min_updated = true;

    m_bike_trainer_data.instantaneous_power_watts = 0;
    m_bike_trainer_data.instantaneous_power_watts_updated = true;

    m_bike_trainer_data.accumulated_power_watts = 0;
    m_bike_trainer_data.accumulated_power_watts_updated = true;
}

/**@brief Function for handling BSP events
 * Runs in app context.
 */
static void bsp_event_handler(bsp_event_t evt)
{
    gfit_lib_return_t ret;

    switch ( evt )
    {
    case BSP_EVENT_KEY_0:
    {
        // Button A = Start
        switch ( m_gfit_state )
        {
        case state_ready:
        {
            NRF_LOG_INFO("Start workout. Set G.FIT state IN USE\r\n");
            ret = gfit_set_state_inuse();
            APP_ERROR_CHECK_BOOL( GFIT_SUCCESS == ret.gfit_response_code );
            m_gfit_state = ret.current_state;
            break;
        }

        case state_finished:
        {
            NRF_LOG_INFO("Resume workout. Set G.FIT state IN USE\r\n");
            ret = gfit_set_state_inuse();
            APP_ERROR_CHECK_BOOL( GFIT_SUCCESS == ret.gfit_response_code );
            m_gfit_state = ret.current_state;
            break;
        }

        case state_inuse:
        {
            // Starting while IN USE is not a valid operation, so we will
            // reuse the button and use it to send a manufacturer specific page
            // over ANT+ FE-C
            // Just as an example, send a counter that is updated every time the button
            // is pushed
            NRF_LOG_INFO("Send manufacturer specific page\r\n");
            m_counter++;
            m_payload[7] = m_counter;
            ret = gfit_fep_send_manufacturer_specific_page(m_payload);
            APP_ERROR_CHECK_BOOL( GFIT_SUCCESS == ret.gfit_response_code );
            m_gfit_state = ret.current_state;
            break;
        }
        case state_off:
        default:
        {
            NRF_LOG_ERROR("Invalid button on this state\r\n");
            break;
        }
        }
        break;
    }

    case BSP_EVENT_KEY_1:
    {
        // Button B = Stop
        switch ( m_gfit_state )
        {
        case state_inuse:
        {
            NRF_LOG_INFO("Pause workout. Set G.FIT state FINISHED\r\n");
            ret = gfit_set_state_finished();
            APP_ERROR_CHECK_BOOL( GFIT_SUCCESS == ret.gfit_response_code );
            m_gfit_state = ret.current_state;
            break;
        }

        case state_finished:
        {
            NRF_LOG_INFO("End workout. Set G.FIT state READY\r\n");
            ret = gfit_set_state_ready();
            APP_ERROR_CHECK_BOOL( GFIT_SUCCESS == ret.gfit_response_code );
            m_gfit_state = ret.current_state;
            fe_fields_reset();
            fe_fields_set();
            break;
        }

        case state_ready: // Intentional fall thru
        case state_off:
        default:
        {
            NRF_LOG_ERROR("Invalid button on this state\r\n");
            break;
        }
        }
        break;
    }

    case BSP_EVENT_KEY_2: // Button C
    {
        // Button C = Disconnect
        if ( state_off != m_gfit_state )
        {
            NRF_LOG_INFO("Disconnect paired HRM\r\n");
            ret = gfit_hrp_disconnect_device();
            APP_ERROR_CHECK_BOOL( GFIT_SUCCESS == ret.gfit_response_code );
            m_gfit_state = ret.current_state;
        }
        break;
    }

    default:
    {
        // Do nothing
        break;
    }
    }
}


/**@brief Function for handling timer events
 */
static void timer_event_handler(void * p_context)
{
    switch ( m_gfit_state )
    {
    case state_inuse:
    {
        m_bike_trainer_data.elapsed_time_seconds++;
        m_bike_trainer_data.elapsed_time_seconds_updated = true;

        m_bike_trainer_data.total_distance_meters += SIM_BIKE_SPEED;                                // Speed is in m/s, this function gets called every second
        m_bike_trainer_data.total_distance_meters_updated = true;

        m_bike_trainer_data.instantaneous_speed_millimeters_per_second = SIM_BIKE_SPEED * 1000;     // Convert to mm/s
        m_bike_trainer_data.instantaneous_speed_millimeters_per_second_updated = true;

        m_bike_trainer_data.instantaneous_cadence_half_revolutions_per_min = SIM_BIKE_CADENCE * 2;  // Convert to half revolutions per minute
        m_bike_trainer_data.instantaneous_cadence_half_revolutions_per_min_updated = true;

        m_bike_trainer_data.instantaneous_power_watts = SIM_BIKE_POWER;
        m_bike_trainer_data.instantaneous_power_watts_updated = true;

        m_bike_trainer_data.accumulated_power_watts += SIM_BIKE_POWER;
        m_bike_trainer_data.accumulated_power_watts_updated = true;
        break;
    }

    case state_finished:
    {
        // Continue showing instantaneous fields, do not increase distance or time
        break;
    }

    case state_ready:
    case state_off:
    default:
    {
        fe_fields_reset();   // Reset values while machine not in use
        break;
    }
    }

    fe_fields_set();
}


/**@brief Function for the timer and BSP initialization
 *
 */
static void utils_init(void)
{
    uint32_t err_code;

    err_code = app_timer_init();
    APP_ERROR_CHECK(err_code);

    err_code = bsp_init(BSP_INIT_LED | BSP_INIT_BUTTONS, bsp_event_handler);
    APP_ERROR_CHECK(err_code);

    err_code = app_timer_create(&m_timer_id, APP_TIMER_MODE_REPEATED, timer_event_handler);
    APP_ERROR_CHECK(err_code);
}

/**
 * @brief Main
 */
int main()
{
    uint32_t err_code;
    gfit_lib_return_t ret;
    uint32_t fe_device_id;

    // Initialize RTT logging
    err_code = NRF_LOG_INIT(NULL);
    APP_ERROR_CHECK(err_code);

    // Initialize the soft device
    softdevice_init();

    // Use DC-DC Converter (recommended to optimize power consumption on D52 module)
    err_code = sd_power_dcdc_mode_set(DC_TO_DC_ON);
    APP_ERROR_CHECK(err_code);

    // Initialize the scheduler
    APP_SCHED_INIT(SCHED_MAX_EVENT_DATA_SIZE, SCHED_QUEUE_SIZE);

    // Initialize buttons and timers
    utils_init();

    // Read the UICR ANT ID (to be used as the fe_device_id)
    if(GFIT_ANT_UICR_DEVICE_NUMBER != 0xFFFFFFFF) // 0xFFFFFFFF indicates likely the ANT ID was not set
    {
        fe_device_id = GFIT_ANT_UICR_DEVICE_NUMBER;
    }
    else if((uint32_t)(NRF_UICR->CUSTOMER[1]) != 0xFFFFFFFF) // If ANT ID was not set, use ESN
    {
        fe_device_id = (uint32_t)(NRF_UICR->CUSTOMER[1]);
    }
    else // If ANT ID and ESN both weren't set, use some default device ID.
    {
        fe_device_id = 0x87654321;
    }

    // Initialize G.FIT Library
    ret = gfit_init(gfit_custom_event_handler, app_error_handler, fe_device_id, GFIT_EVALUATION_LICENSE_KEY);
    APP_ERROR_CHECK_BOOL( GFIT_SUCCESS == ret.gfit_response_code );

    m_gfit_state = ret.current_state;

    NRF_LOG_INFO("Initialized G.FIT. Device ID: %lu\r\n", fe_device_id);

    // Configure G.FIT library
    // Set pairing mode to proximity
    ret = gfit_hrp_set_pairing_mode(pairing_mode_proximity);
    APP_ERROR_CHECK_BOOL( GFIT_SUCCESS == ret.gfit_response_code );

    // Configure proximity pairing settings
    ret = gfit_hrp_set_pairing_proximity(DEMO_SEARCH_THRESHOLD, DEMO_PAIR_THRESHOLD);
    APP_ERROR_CHECK_BOOL( GFIT_SUCCESS == ret.gfit_response_code );

    // Configure fitness equipment type
    ret = gfit_fep_set_equipment_type(bike);
    APP_ERROR_CHECK_BOOL( GFIT_SUCCESS == ret.gfit_response_code );

    // Configure supported FE fields to be transmitted over ANT+ FE-C and BLE FMTS for given fitness equipment type
    fe_fields_init();
    fe_fields_set();

    NRF_LOG_INFO("Configured G.FIT\r\n");

    // Set state to READY
    ret = gfit_set_state_ready();
    APP_ERROR_CHECK_BOOL( GFIT_SUCCESS == ret.gfit_response_code );

    m_gfit_state = ret.current_state;

    NRF_LOG_INFO("Set G.FIT state READY\r\n");

    // Start timer for simulated FE-data
    err_code = app_timer_start(m_timer_id, TIMER_TICKS, NULL);
    APP_ERROR_CHECK(err_code);

    // loop forever
    while (1)
    {
        app_sched_execute();

        NRF_LOG_FLUSH();

        err_code = sd_app_evt_wait();
        APP_ERROR_CHECK(err_code);
    }
}
