/*
 This software is subject to the license described in the license.txt file included with this software distribution. You may not use this file except in compliance with this license.
 Copyright © Dynastream Innovations Inc. 2017
 All rights reserved.
 */

import UIKit
import CoreBluetooth

class DataCard: Equatable {

    let DEFAULT_VALUE = "---"
    let title: String
    let units: String
    let characteristicUUID: CBUUID
    let serviceUUID: CBUUID
    let decoder: (Data) -> (Any)?
    let forDisplay: Bool
    let recordMessageType: FIT_MESSAGE_TYPE?
    let sessionMessageType: FIT_MESSAGE_TYPE?
    var value: Float? = nil
    var valueString: String
    var cell: DataCardCell?

    init(forServiceUUID serviceUUID: CBUUID, forCharacteristicUUID characteristicUUID: CBUUID, titled title: String, withUnits units: String, decodeWith decoder: @escaping (Data) -> (Any), withFitRecordMessageType fitRecordMessageType: FIT_MESSAGE_TYPE, withFitSessionMessageType fitSessionMessageType: FIT_MESSAGE_TYPE, forDisplay display: Bool = true) {
        self.title = title
        self.units = units
        self.serviceUUID = serviceUUID
        self.characteristicUUID = characteristicUUID
        self.decoder = decoder
        self.forDisplay = display
        self.valueString = DEFAULT_VALUE
        self.recordMessageType = fitRecordMessageType
        self.sessionMessageType = fitSessionMessageType
    }

    init(fromDataField dataField: ServiceHierarchy.DataField,forServiceUUID serviceUUID: CBUUID, forCharacteristicUUID characteristicUUID: CBUUID) {
        self.title = dataField.title
        self.units = dataField.units
        self.decoder = dataField.decoder
        self.forDisplay = dataField.forDisplay
        self.recordMessageType = dataField.fitRecordMessageType
        self.sessionMessageType = dataField.fitSessionMessageType
        self.serviceUUID = serviceUUID
        self.characteristicUUID = characteristicUUID
        self.valueString = DEFAULT_VALUE
    }

    func updateCard(withRawData data: Data?) {
        if data != nil {
            self.updateCard(withValue: decoder(data!)!)
        }
    }

    func updateCard(withValue inValue: Any) {
        if inValue as? String == DEFAULT_VALUE {
            return
        }
        self.valueString = buildValueString(withValue: inValue)
        if Float(String(describing: inValue)) != nil {
            self.value = Float(String(describing: inValue))
        }
        cell?.update(fromCard: self)
    }

    func setCell(_ cell: DataCardCell) {
        self.cell = cell
        cell.titleLabel.text = self.title
        cell.update(fromCard: self)
    }

    static func ==(LHS: DataCard, RHS: DataCard) -> Bool {
        return LHS.title == RHS.title && LHS.characteristicUUID == RHS.characteristicUUID
    }

    private func buildValueString(withValue value: Any) -> String {
        return String(describing: value) + self.units
    }
}
