/*
 This software is subject to the license described in the license.txt file included with this software distribution. You may not use this file except in compliance with this license.
 Copyright © Dynastream Innovations Inc. 2017
 All rights reserved.
*/

import UIKit
import CoreBluetooth

class DataManager: NSObject, UICollectionViewDataSource {

    var displayDataCards: [DataCard] = []
    var metaDataCards: [DataCard] = []
    var delegate: DataManagerDelegate

    init(withDelegate delegate: DataManagerDelegate) {
        self.delegate = delegate
        super.init()
    }

    func addDataCard(_ card: DataCard) {
        if allDataCards().contains(card) {
            return
        }
        self.delegate.readData(forServiceUUID: card.serviceUUID, forCharachteristicUUID: card.characteristicUUID)
        if card.forDisplay {
            self.displayDataCards.append(card)
        } else {
            self.metaDataCards.append(card)
        }
    }

    func updateDataCard(withRawData data: Data?, forCardWithCharachterisitcUUID charachteristicUUID: CBUUID){
        for card in allDataCards() {
            if card.characteristicUUID == charachteristicUUID {
                card.updateCard(withRawData: data)
            }
        }
    }

    func getFitRecordValues() -> [FitTypeValuePair] {
        var records: [FitTypeValuePair] = []
        for card in displayDataCards {
            if cardIsNotForFitRecord(card) {
                continue
            }
            records.append(FitTypeValuePair(value: (card.value)! , messageType: card.recordMessageType!))
        }
        return records
    }

    func getFitSessionValues() -> [FitTypeValuePair] {
        var sessionMessages:[FitTypeValuePair] = []
        for card in displayDataCards {
            if cardIsNotForFitSession(card) {
                continue
            }
            sessionMessages.append(FitTypeValuePair(value: (card.value)!, messageType: card.sessionMessageType!))
        }
        return sessionMessages
    }

    func numberOfSections(in collectionView: UICollectionView) -> Int {
        return 1
    }

    func collectionView(_ collectionView: UICollectionView, numberOfItemsInSection section: Int) -> Int {
        return displayDataCards.count
    }

    func collectionView(_ collectionView: UICollectionView, cellForItemAt indexPath: IndexPath) -> UICollectionViewCell {
        guard let cell = collectionView.dequeueReusableCell(withReuseIdentifier: "DataCardCell", for: indexPath) as? DataCardCell else {
            fatalError("Cell is not an instance of DataCardCollectionViewCell.")
        }
        displayDataCards[indexPath.row].setCell(cell)
        cell.layer.cornerRadius = cell.frame.width / 6
        cell.layer.masksToBounds = true
        cell.backgroundColor = UserInterfaceElements.cardColor
        return cell
    }

    private func cardIsNotForFitRecord(_ card: DataCard) -> Bool {
        return card.recordMessageType == nil || card.value == nil
    }

    private func cardIsNotForFitSession(_ card: DataCard) -> Bool {
        return card.sessionMessageType == nil || card.value == nil
    }

    private func allDataCards() -> [DataCard] {
        return self.displayDataCards + self.metaDataCards
    }
}
