/*
 This software is subject to the license described in the license.txt file included with this software distribution. You may not use this file except in compliance with this license.
 Copyright © Dynastream Innovations Inc. 2017
 All rights reserved.
 */

import UIKit
import CoreBluetooth

class DataViewController: UIViewController, UINavigationControllerDelegate, UICollectionViewDelegate, DeviceConnectionManagerDelegate, DataManagerDelegate, FitFileManagerDelegate, UICollectionViewDelegateFlowLayout {

    var deviceConnectionManager: DeviceConnectionManager?
    var dataManager: DataManager?
    var fitFileManager: FitFileManager?

    @IBOutlet var dataCollectionView: UICollectionView!
    @IBOutlet weak var navigationTitle: UINavigationItem!

    @IBAction func backToTableButton(_ sender: UIBarButtonItem) {
        let saveAlert = buildSaveAlert()
        self.present(saveAlert, animated: true, completion: nil)
    }

    required init?(coder aDecoder: NSCoder) {
        super.init(coder: aDecoder)
        self.fitFileManager = FitFileManager(self)
        self.dataManager = DataManager(withDelegate: self)
        self.deviceConnectionManager = DeviceConnectionManager(withDelegate: self)
    }

    func saveAndQuitReording(forAlertAction action: UIAlertAction) -> Void {
        deviceConnectionManager?.disconnect()
        self.fitFileManager?.stopRecording()
        self.fitFileManager?.completeSession(withName: buildFileTitle())
        navigationController?.popViewController(animated: true)
    }

    func quitRecording(forAlertAction action: UIAlertAction) -> Void {
        deviceConnectionManager?.disconnect()
        self.fitFileManager?.stopRecording()
        navigationController?.popViewController(animated: true)
    }

    override func viewDidLoad() {
        super.viewDidLoad()
        self.dataCollectionView.delegate = self
        self.dataCollectionView.dataSource = self.dataManager
        self.deviceConnectionManager?.connect()
        self.fitFileManager?.startRecording()
        self.navigationTitle.title = "G.FIT"
    }

    func connectionNotified(withData data: Data?, forCharacteristicUUID characteristicUUID: CBUUID) {
        self.dataManager?.updateDataCard(withRawData: data, forCardWithCharachterisitcUUID: characteristicUUID)
    }

    func connectionHasCharacteristics(_ characteristics: [CBCharacteristic], forService service: CBService){
        for characteristic in characteristics {
            addDataField(for: service, with: characteristic)
            setNavigationTitle(basedOn: characteristic)
        }
        self.dataCollectionView.reloadData()
    }

    func collectionView(_ collectionView: UICollectionView, layout collectionViewLayout: UICollectionViewLayout, sizeForItemAt indexPath: IndexPath) -> CGSize {
        let maxCellWidth = CGFloat(300)
        let itemsPerRow = max(2, ceil(view.frame.width/maxCellWidth))
        let paddingWidth = sectionInsets().left * (itemsPerRow + 1)
        let availableWidth = view.frame.width - paddingWidth
        let widthPerItem = availableWidth / itemsPerRow
        return CGSize(width: widthPerItem, height: widthPerItem)
    }

    func collectionView(_ collectionView: UICollectionView, layout collectionViewLayout: UICollectionViewLayout, insetForSectionAt section: Int) -> UIEdgeInsets {
        return sectionInsets()
    }

    func collectionView(_ collectionView: UICollectionView, layout collectionViewLayout: UICollectionViewLayout, minimumLineSpacingForSectionAt section: Int) -> CGFloat {
        return sectionInsets().left
    }

    func readData(forServiceUUID serviceUUID: CBUUID, forCharachteristicUUID charachteristicUUID: CBUUID){
        self.deviceConnectionManager?.readCharachteristic(withServiceUUID: serviceUUID, withCharacteristicUUID: charachteristicUUID)
    }

    func fitRequestedRecordMessage() {
        let records = self.dataManager?.getFitRecordValues()
        let recordsPointer = UnsafeMutablePointer<FitTypeValuePair>.allocate(capacity: (records?.count)!)
        recordsPointer.initialize(from: records!, count: (records?.count)!)
        self.fitFileManager?.addRecordMessage(recordsPointer, numberOfSamples: Int32(records!.count))
        recordsPointer.deinitialize()
    }

    func fitRequestedSessionMessages() {
        let sessionMessages = self.dataManager?.getFitSessionValues()
        let sessionPointer = UnsafeMutablePointer<FitTypeValuePair>.allocate(capacity: (sessionMessages?.count)!)
        sessionPointer.initialize(from: sessionMessages!, count: (sessionMessages?.count)!)
        self.fitFileManager?.addSessionMessage(sessionPointer, numberOfSamples: Int32(sessionMessages!.count))
        sessionPointer.deinitialize()
    }

    override func prepare(for segue: UIStoryboardSegue, sender: Any?) {
        if segue.identifier == "ShowDeviceData" {
            setupDeviceInformationDestination(fromSegue: segue)
        } else if segue.identifier == "BackToTable" {

        } else {
                fatalError("Unexpected destination: \(segue.destination)")
        }
    }

    private func setupDeviceInformationDestination(fromSegue segue: UIStoryboardSegue) {
        if let destination = segue.destination as? DeviceInfoViewController {
            if connectedDeviceSupportsDeviceInformationService() {
                cardIteration: for infoCard in (self.dataManager?.metaDataCards)! {
                    switch infoCard.title {
                    case "Manufacturer Name":
                        destination.manufacturerNameValue = String(describing: infoCard.valueString)
                        continue cardIteration
                    case "Model Number":
                        destination.modelNumberValue = String(describing: infoCard.valueString)
                        continue cardIteration
                    case "Firmware Revision":
                        destination.firmwareRevisionValue = String(describing: infoCard.valueString)
                        continue cardIteration
                    case "Software Revision":
                        destination.softwareRevisionValue = String(describing: infoCard.valueString)
                        continue cardIteration
                    default:
                        continue cardIteration
                    }
                }
            }
        }
    }

    private func connectedDeviceSupportsDeviceInformationService() -> Bool {
        let connectedDeviceSupportedServices = self.deviceConnectionManager?.connectedDevice?.services
        let deviceInfoIsSupported = connectedDeviceSupportedServices?.contains(where: {service in return service.uuid == ServiceHierarchy.deviceInformationService.uuid})
        return deviceInfoIsSupported != nil
    }

    private func addDataField(for service: CBService, with characteristic: CBCharacteristic) {
        for field in ServiceHierarchy.lookupDatafields(forCharacteristicUUID: characteristic.uuid) {
            let dataCard = DataCard(fromDataField: field, forServiceUUID: service.uuid, forCharacteristicUUID: characteristic.uuid)
            self.dataManager?.addDataCard(dataCard)
        }
    }

    private func setNavigationTitle(basedOn characteristic: CBCharacteristic) {
        switch characteristic.uuid {
        case ServiceHierarchy.treadmillDataUUID:
            self.navigationTitle.title = "Treadmill"
            break
        case ServiceHierarchy.indoorBikeDataUUID:
            self.navigationTitle.title = "Bike"
            break
        case ServiceHierarchy.stepClimberDataUUID:
            self.navigationTitle.title = "Climber"
            break
        case ServiceHierarchy.rowerDataUUID:
            self.navigationTitle.title = "Rower"
            break
        case ServiceHierarchy.crossTrainerDataUUID:
            self.navigationTitle.title = "Elliptical"
            break
        default:
            break
        }
    }

    private func buildSaveAlert() -> UIAlertController {
        let saveAlert = UIAlertController(title: nil, message: nil, preferredStyle: .actionSheet)
        let saveAction = UIAlertAction(title: "Save Session Data", style: .default, handler: self.saveAndQuitReording)
        let deleteAction = UIAlertAction(title: "Delete Data", style: .destructive, handler: self.quitRecording)
        let cancelAction = UIAlertAction(title: "Cancel", style: .cancel, handler: nil)
        saveAlert.addAction(saveAction)
        saveAlert.addAction(cancelAction)
        saveAlert.addAction(deleteAction)
        return saveAlert
    }

    private func buildFileTitle() -> String {
        let titlePrefix = (self.navigationTitle.title ?? "G.FIT") + "_"
        let dateFormatter = DateFormatter()
        dateFormatter.dateFormat = "yyyy-MM-dd_hh-mm-ss"
        return titlePrefix + dateFormatter.string(from: Date())
    }

    private func sectionInsets() -> UIEdgeInsets {
        return UIEdgeInsets(top: 10.0, left: 10.0, bottom: 10.0, right: 10.0)
    }
}
