/*
 This software is subject to the license described in the license.txt file included with this software distribution. You may not use this file except in compliance with this license.
 Copyright © Dynastream Innovations Inc. 2017
 All rights reserved.
 */

import Foundation

// Describe how to decode recieved data for each data field
class Decoders {
    // Flags schemes used for decoding FTMS data
    private static let treadmillFlagsScheme =       [2, 2, 2, 3, 4, 4, 1, 1, 5, 1, 1, 2, 2, 4]
    private static let stationaryBikeFlagsScheme =  [2, 2, 2, 2, 2, 3, 2, 2, 2, 5, 1, 1, 2, 2]
    private static let rowerFlagsScheme =           [2, 3, 1, 3, 2, 2, 2, 2, 2, 5, 1, 1, 2, 2]
    private static let crossTrainerFlagsScheme =    [3, 2, 2, 3, 4, 2, 4, 4, 2, 2, 2, 5, 1, 1, 2, 2]
    private static let stepClimberFlagScheme =      [2, 4, 2, 2, 2, 5, 1, 1, 2, 2]

    class func decodeUTF8s(withData data: Data) -> String {
        let array = Decoders.dataToByteArray(data)
        return NSString(bytes: array, length: array.count, encoding: String.Encoding.utf8.rawValue)! as String
    }

    class func decodeHRValue(withData data: Data) -> Int {
        let array = Decoders.dataToByteArray(data)
        var bpmValue : Int = 0;
        if ((array[0] & 0x01) == 0) {
            bpmValue = Int(array[1])
        } else {
            //Convert Endianess
            bpmValue = Int(UInt16(array[2]) << 0x08 + UInt16(array[1]))
        }
        return bpmValue
    }

    class func decodeBatteryStatus(withData data: Data) -> Int {
        return Int(Decoders.dataToByteArray(data)[0])
    }


    class func decodeHRLocation(withData data:Data) -> String {
        let location = (data as NSData).bytes.bindMemory(to: UInt16.self, capacity: data.count)
        switch (location[0]) {
        case 0:
            return "Other"
        case 1:
            return "Chest"
        case 2:
            return "Wrist"
        case 3:
            return "Finger"
        case 4:
            return "Hand"
        case 5:
            return "Ear Lobe"
        case 6:
            return "Foot"
        default:
            return "Invalid"
        }
    }

    class func decodeFtmsFeatures(withData data: Data) -> String {
        let array = Decoders.dataToByteArray(data)
        if array[0] & UInt8(0x01 << 0) != 0 { print("Average Speed Supported") }
        if array[0] & UInt8(0x01 << 1) != 0 { print("Cadence Supported") }
        if array[0] & UInt8(0x01 << 2) != 0 { print("Total Distance Supported") }
        if array[0] & UInt8(0x01 << 3) != 0 { print("Inclination Supported") }
        if array[0] & UInt8(0x01 << 4) != 0 { print("Elevation Gain Supported") }
        if array[0] & UInt8(0x01 << 5) != 0 { print("Pace Supported") }
        if array[0] & UInt8(0x01 << 6) != 0 { print("Step Count Supported") }
        if array[0] & UInt8(0x01 << 7) != 0 { print("Resistance Level Supported") }
        if array[2] & UInt8(0x01 << 0) != 0 { print("Stride Count Supported") }
        if array[2] & UInt8(0x01 << 1) != 0 { print("Expended Energy Supported") }
        if array[2] & UInt8(0x01 << 2) != 0 { print("Heart Rate Measurement Supported") }
        if array[2] & UInt8(0x01 << 3) != 0 { print("Metabolic Equivalent Supported") }
        if array[2] & UInt8(0x01 << 4) != 0 { print("Elapsed Time Supported") }
        if array[2] & UInt8(0x01 << 5) != 0 { print("Remaining Time Supported") }
        if array[2] & UInt8(0x01 << 6) != 0 { print("Power Measurement Supported") }
        if array[2] & UInt8(0x01 << 7) != 0 { print("Force on Belt and Power Output Supported") }
        if array[3] & UInt8(0x01 << 0) != 0 { print("User Data Retention Supported") }
        return ""
    }

    // Treadmill Decoders
    class func decodeTreadmillElapsedTime(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: treadmillFlagsScheme, forFlagBit: 10, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeTreadmillInstantaneousPace(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: treadmillFlagsScheme, forFlagBit: 5, forByteWidth: 1) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeTreadmillPositiveElevationGain(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: treadmillFlagsScheme, forFlagBit: 4, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue)/10, withFormatString: "%0.1f")
        }
        return "---"
    }

    class func decodeTreadmillNegativeElevationGain(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: treadmillFlagsScheme, forFlagBit: 4, forByteWidth: 2, withByteOffset: 2) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue)/10, withFormatString: "%0.1f")
        }
        return "---"
    }

    class func decodeTreadmillInclination(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: treadmillFlagsScheme, forFlagBit: 3, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(Int16(truncatingBitPattern: decodedValue))/10, withFormatString: "%0.1f")
        }
        return "---"
    }

    class func decodeTreadmillTotalDistance(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: treadmillFlagsScheme, forFlagBit: 2, forByteWidth: 3) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeTreadmillInstantaneousSpeed(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: treadmillFlagsScheme, forFlagBit: 0, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue)/100, withFormatString: "%0.2f")
        }
        return "---"
    }

    class func decodeTreadmillMetabolicEquivalent(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: treadmillFlagsScheme, forFlagBit: 9, forByteWidth: 1) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue)/10, withFormatString: "%0.1f")
        }
        return "---"
    }

    class func decodeTreadmillTotalEnergy(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: treadmillFlagsScheme, forFlagBit: 7, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeTreadmillEnergyPerHour(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: treadmillFlagsScheme, forFlagBit: 7, forByteWidth: 2, withByteOffset: 2) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeTreadmillEnergyPerMinute(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: treadmillFlagsScheme, forFlagBit: 7, forByteWidth: 1, withByteOffset: 4) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    // Stationary Bike Decoders
    class func decodeStationaryBikeElapsedTime(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: stationaryBikeFlagsScheme, forFlagBit: 11, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeStationaryBikeInstantaneousSpeed(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: treadmillFlagsScheme, forFlagBit: 0, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue)/100, withFormatString: "%0.2f")
        }
        return "---"
    }

    class func decodeStationaryBikeResistanceLevel(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: stationaryBikeFlagsScheme, forFlagBit: 5, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(Int16(truncatingBitPattern: decodedValue)), withFormatString: "%0.0f")
        }
        return "---"
    }


    class func decodeStationaryBikeInstantaneousCadence(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: stationaryBikeFlagsScheme, forFlagBit: 1, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue)/2, withFormatString: "%0.1f")
        }
        return "---"
    }


    class func decodeStationaryBikeInstantaneousPower(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: stationaryBikeFlagsScheme, forFlagBit: 6, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(Int16(truncatingBitPattern: decodedValue)), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeStationaryBikeTotalDistance(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: stationaryBikeFlagsScheme, forFlagBit: 4, forByteWidth: 3) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeStationaryBikeMetabolicEquivalent(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: stationaryBikeFlagsScheme, forFlagBit: 10, forByteWidth: 1) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue)/10, withFormatString: "%0.1f")
        }
        return "---"
    }

    class func decodeStationaryBikeTotalEnergy(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: stationaryBikeFlagsScheme, forFlagBit: 8, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeStationaryBikeEnergyPerMinute(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: stationaryBikeFlagsScheme, forFlagBit: 8, forByteWidth: 1, withByteOffset: 4) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeStationaryBikeEnergyPerHour(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: stationaryBikeFlagsScheme, forFlagBit: 8, forByteWidth: 2, withByteOffset: 2) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    // Rower Decoders
    class func decodeRowerElapsedTime(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: rowerFlagsScheme, forFlagBit: 11, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeRowerResistanceLevel(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: rowerFlagsScheme, forFlagBit: 7, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(Int16(truncatingBitPattern: decodedValue)), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeRowerStrokeCount(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: rowerFlagsScheme, forFlagBit: 0, forByteWidth: 2, withByteOffset: 1) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeRowerStrokeRate(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: rowerFlagsScheme, forFlagBit: 0, forByteWidth: 1) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue)/2, withFormatString: "%0.1f")
        }
        return "---"
    }

    class func decodeRowerInstantaneousPower(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: rowerFlagsScheme, forFlagBit: 5, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(Int16(truncatingBitPattern: decodedValue)), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeRowerMetabolicEquivalent(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: rowerFlagsScheme, forFlagBit: 10, forByteWidth: 1) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue)/10, withFormatString: "%0.1f")
        }
        return "---"
    }

    class func decodeRowerTotalEnergy(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: rowerFlagsScheme, forFlagBit: 8, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeRowerEnergyPerHour(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: rowerFlagsScheme, forFlagBit: 8, forByteWidth: 2, withByteOffset: 2) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeRowerEnergyPerMinute(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: rowerFlagsScheme, forFlagBit: 8, forByteWidth: 1, withByteOffset: 4) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeRowerTotalDistance(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: rowerFlagsScheme, forFlagBit: 2, forByteWidth: 3) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    // Cross Trainer Decoders
    class func decodeCrossTrainerElapsedTime(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: crossTrainerFlagsScheme, forFlagBit: 13, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeCrossTrainerInstantaneousSpeed(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: crossTrainerFlagsScheme, forFlagBit: 0, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue)/100, withFormatString: "%0.2f")
        }
        return "---"
    }

    class func decodeCrossTrainerResistanceLevel(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: crossTrainerFlagsScheme, forFlagBit: 7, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(Int16(truncatingBitPattern: decodedValue))/10, withFormatString: "%0.1f")
        }
        return "---"
    }

    class func decodeCrossTrainerInclination(withData data: Data) -> String {

        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: crossTrainerFlagsScheme, forFlagBit: 6, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(Int16(truncatingBitPattern: decodedValue))/10, withFormatString: "%0.1f")
        }
        return "---"
    }

    class func decodeCrossTrainerPositiveElevationGain(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: crossTrainerFlagsScheme, forFlagBit: 5, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeCrossTrainerStrideCount(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: crossTrainerFlagsScheme, forFlagBit: 4, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue)/10, withFormatString: "%0.1f")
        }
        return "---"
    }

    class func decodeCrossTrainerCadence(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: crossTrainerFlagsScheme, forFlagBit: 3, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeCrossTrainerInstantaneousPower(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: crossTrainerFlagsScheme, forFlagBit: 8, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(Int16(truncatingBitPattern: decodedValue)), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeCrossTrainerMetabolicEquivalent(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: crossTrainerFlagsScheme, forFlagBit: 12, forByteWidth: 1) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue)/10, withFormatString: "%0.1f")
        }
        return "---"
    }

    class func decodeCrossTrainerTotalEnergy(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: crossTrainerFlagsScheme, forFlagBit: 10, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeCrossTrainerEnergyPerHour(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: crossTrainerFlagsScheme, forFlagBit: 10, forByteWidth: 2, withByteOffset: 2) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeCrossTrainerEnergyPerMinute(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: crossTrainerFlagsScheme, forFlagBit: 10, forByteWidth: 1, withByteOffset: 4) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeCrossTrainerTotalDistance(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: crossTrainerFlagsScheme, forFlagBit: 2, forByteWidth: 3) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    // Step Climber Decoders
    class func decodeStepClimberElapsedTime(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: stepClimberFlagScheme, forFlagBit: 7, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeStepClimberStepsClimbedCount(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: stepClimberFlagScheme, forFlagBit: 0, forByteWidth: 2, withByteOffset: 2) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeStepClimberCadence(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: stepClimberFlagScheme, forFlagBit: 1, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }
    
    class func decodeStepClimberPositiveElevationGain(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: stepClimberFlagScheme, forFlagBit: 3, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeStepClimberMetabolicEquivalent(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: stepClimberFlagScheme, forFlagBit: 6, forByteWidth: 1) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue)/10, withFormatString: "%0.1f")
        }
        return "---"
    }

    class func decodeStepClimberTotalEnergy(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: stepClimberFlagScheme, forFlagBit: 4, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeStepClimberEnergyPerHour(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: stepClimberFlagScheme, forFlagBit: 4, forByteWidth: 2, withByteOffset: 2) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeStepClimberEnergyPerMinute(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: stepClimberFlagScheme, forFlagBit: 4, forByteWidth: 1, withByteOffset: 4) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    class func decodeStepClimberFloorsClimbed(withData data: Data) -> String {
        if let decodedValue = decodeFTMSData(withData: data, withFlagsDataMap: stepClimberFlagScheme, forFlagBit: 0, forByteWidth: 2) {
            return Decoders.formatDisplayString(forValue: Double(decodedValue), withFormatString: "%0.0f")
        }
        return "---"
    }

    //Decode data sent over the fitness machinde BLE service
    // -data                 Raw data passed in from the peripheral
    // -flagDataMap          Numbers mapping each flag to the number of bytes in the field it represents
    // -flagBit              The flag the correspondes to the data field that is required
    // -numberOfBytes        The number of bytes of data required
    // -byteOffset           The number of bytes from the start of the field to the start of the data section required
    private class func decodeFTMSData(withData data: Data, withFlagsDataMap flagsDataMap: [Int], forFlagBit flagBit: Int, forByteWidth numberOfBytes: Int, withByteOffset byteOffset: Int = 0) -> UInt32?{
        let array = Decoders.dataToByteArray(data)
        if let prefixByteCount = Decoders.calculatePrefixByteCount(fromArray: array, forFlagBit: flagBit, forDataMap: flagsDataMap, withOffset: byteOffset) {
            return Decoders.extractData(fromArray: array, withDataSize: numberOfBytes, withPrefixSize: prefixByteCount)
        }
        return nil
    }

    // MARK: Helper functions
    private static let MORE_DATA_BIT = 0
    private static let MORE_DATA_SIZE_IDX = 1
    private static let MORE_DATA_BIT_MASK = UInt32(0x01)
    private static let FLAG_CLEAR = UInt32(0)
    private static let FLAG_SET = UInt32(1)
    private static let FLAGS_SIZE_IDX = 0

    class func calculatePrefixByteCount(fromArray array: [UInt8], forFlagBit flagBit: Int, forDataMap flagsDataMap: [Int], withOffset byteOffset: Int) -> Int? {
        let flags = getFlagsField(fromeArray: array, withFlagsSize: flagsDataMap[FLAGS_SIZE_IDX])
        var prefixByteCount: Int = flagsDataMap[FLAGS_SIZE_IDX] + byteOffset

        if dataNotPresent(forFlagBit: flagBit, withFlags: flags) {
            return nil
        }
        else {
            if flagBit != MORE_DATA_BIT {
                prefixByteCount += flagsDataMap[MORE_DATA_SIZE_IDX]
            }
            for bit in 0..<flagBit {
                if flagIsSet(bit: bit, flags: flags) {
                    prefixByteCount += flagsDataMap[bit + 1]
                }
            }
        }
        return prefixByteCount
    }

    class func dataToByteArray(_ data: Data) -> [UInt8] {
        let count = data.count / MemoryLayout<UInt8>.size
        var array = [UInt8](repeating: 0, count: count)
        (data as NSData).getBytes(&array, length:count * MemoryLayout<Int8>.size)
        return array
    }

    class func extractData(fromArray array: [UInt8], withDataSize numberOfBytes: Int, withPrefixSize prefixByteCount: Int) -> UInt32 {
        var fullValue = UInt32(0)
        for byteNumber in 0..<numberOfBytes {
            fullValue += UInt32(array[prefixByteCount + byteNumber]) << UInt32(byteNumber*8)
        }
        return fullValue
    }

    private class func getFlagsField(fromeArray array: [UInt8], withFlagsSize flagsSize: Int) -> UInt32 {
        var flags = UInt32(0)
        for byteNumber in 0..<flagsSize {
            flags += UInt32(array[byteNumber]) << UInt32(byteNumber*8)
        }
        return flags
    }

    private class func dataNotPresent(forFlagBit flagBit: Int, withFlags flags: UInt32) -> Bool {
        return Decoders.isMoreDataBitAndNotPresent(forFlagBit: flagBit, withFlags: flags) || Decoders.fieldNotPresent(forFlagBit: flagBit, withFlags: flags)
    }

    private class func isMoreDataBitAndNotPresent(forFlagBit flagBit: Int, withFlags flags: UInt32) -> Bool {
        return flagBit == MORE_DATA_BIT && (flags & MORE_DATA_BIT_MASK) == FLAG_SET
    }

    private class func fieldNotPresent(forFlagBit flagBit: Int, withFlags flags: UInt32) -> Bool {
        return flagBit != MORE_DATA_BIT && (flags & UInt32(0x01 << UInt32(flagBit))) == FLAG_CLEAR
    }

    private class func flagIsSet(bit: Int, flags: UInt32) -> Bool {
        return (flags & UInt32(0x01 << bit)) != FLAG_CLEAR
    }

    private class func formatDisplayString(forValue value: Double, withFormatString formatString: String) -> String{
        return String.localizedStringWithFormat(formatString, value)
    }
}
