/*
 This software is subject to the license described in the license.txt file included with this software distribution. You may not use this file except in compliance with this license.
 Copyright © Dynastream Innovations Inc. 2017
 All rights reserved.
 */

import UIKit
import CoreBluetooth

class DeviceConnectionManager: NSObject, CBCentralManagerDelegate, CBPeripheralDelegate {

    var btCentralManager: CBCentralManager?
    var connectedDevice: CBPeripheral?
    let delegate: DeviceConnectionManagerDelegate

    init(withDelegate delegate: DeviceConnectionManagerDelegate) {
        self.delegate = delegate
        super.init()
    }

    func setConnectedDevice(with device: CBPeripheral?) {
        self.connectedDevice = device
    }

    func connect() {
        if capableOfConnecting() {
            btCentralManager?.delegate = self
            connectedDevice?.delegate = self
            btCentralManager?.connect((connectedDevice)!, options: nil)
        }
    }

    func disconnect() {
        if haveConnectedDevice() {
            btCentralManager?.cancelPeripheralConnection(connectedDevice!)
        }
    }

    func readCharachteristic(withServiceUUID serviceUUID: CBUUID, withCharacteristicUUID charcteristicUUID: CBUUID) {
        if connectedDevice != nil {
            for service in (connectedDevice?.services)! {
                if service.uuid == serviceUUID {
                    for characteristic in service.characteristics! {
                        if characteristic.uuid == charcteristicUUID {
                            self.connectedDevice?.readValue(for: characteristic)
                        }
                    }
                }
            }
        }
    }

    func centralManagerDidUpdateState(_ central: CBCentralManager) {
        if connectedDevice != nil && central.state != .poweredOff {
            btCentralManager?.connect(connectedDevice!, options: nil)
        }
    }

    func centralManager(_ central: CBCentralManager, didConnect peripheral: CBPeripheral) {
        if peripheral.identifier == connectedDevice?.identifier {
            peripheral.discoverServices(ServiceHierarchy.services.allKeys as? [CBUUID])
        }
    }

    func peripheral(_ peripheral: CBPeripheral, didDiscoverServices error: Error?) {
        if error != nil {
            return
        }
        for service : CBService in peripheral.services! {
            if ServiceHierarchy.services[service.uuid] != nil {
                peripheral.discoverCharacteristics(nil, for: service)
            }
        }
    }

    func peripheral(_ peripheral: CBPeripheral, didDiscoverCharacteristicsFor service: CBService, error: Error?) {
        if error != nil || service.characteristics == nil {
            return
        }
        for characteristic in service.characteristics! {
            self.connectedDevice?.setNotifyValue(true, for: characteristic)
        }
        self.delegate.connectionHasCharacteristics(service.characteristics!, forService: service)
    }

    func peripheral(_ peripheral: CBPeripheral, didUpdateValueFor characteristic: CBCharacteristic, error: Error?) {
        self.delegate.connectionNotified(withData: characteristic.value, forCharacteristicUUID: characteristic.uuid)
    }

    private func capableOfConnecting() -> Bool {
        return connectedDevice != nil && btCentralManager != nil
    }

    private func haveConnectedDevice() -> Bool {
        return self.connectedDevice != nil
    }
}
