/*
 This software is subject to the license described in the license.txt file included with this software distribution. You may not use this file except in compliance with this license.
 Copyright © Dynastream Innovations Inc. 2017
 All rights reserved.
 */

import UIKit
import CoreBluetooth

class DeviceScanManager: NSObject, CBCentralManagerDelegate {

    private let INVALID_RSSI = 127 as NSNumber
    private let MINUMUN_RSSI = -127 as Int
    private var scannedUUIDs: [CBUUID]? = nil
    private var solicitedServices: [CBUUID] = []
    private var allowDuplicates: Bool = true
    let delegate: DeviceScanManagerDelegate
    var rssiThreshold: Int
    var btCentralManager : CBCentralManager?
    var devices: [Device] = []

    init(withDelegate delegate: DeviceScanManagerDelegate) {
        self.delegate = delegate
        self.rssiThreshold = self.MINUMUN_RSSI
        super.init()
        self.btCentralManager = CBCentralManager(delegate: self, queue: nil)
    }

    func openScan(forUUIDs UUIDs: [CBUUID]? = nil, solicitingServices services: [CBUUID] = [], withDuplicates allowDuplicates: Bool = true) {
        self.scannedUUIDs = UUIDs
        self.solicitedServices = services
        self.allowDuplicates = allowDuplicates
        let options = buildScanOptions(forServices: services)
        btCentralManager?.scanForPeripherals(withServices: UUIDs, options: options as? [String : AnyObject])
    }

    func reopenScan() {
        self.btCentralManager = CBCentralManager(delegate: self, queue: nil)
        self.openScan(forUUIDs: self.scannedUUIDs, solicitingServices: self.solicitedServices, withDuplicates: self.allowDuplicates)
    }

    func closeScan() {
        btCentralManager?.stopScan()
    }

    func setRssiThreshod(_ rssiThreshold: Int) {
        self.rssiThreshold = rssiThreshold
    }

    func numberOfDevices() -> Int {
        return self.devices.count
    }

    func getDevice(forIndex index: Int) -> Device {
        return devices[index]
    }

    func updateDevice(_ device: Device) {
        if self.devices.contains(device) {
            let oldDevice = self.devices[self.devices.index(of: device)!]
            oldDevice.signalStrength = device.signalStrength
            oldDevice.updatePrevMsgTime()
        } else {
            self.devices.append(device)
        }
        self.delegate.scannerUpdated()
    }

    func centralManagerDidUpdateState(_ central: CBCentralManager) {
        if central.state != .poweredOff {
            openScan(forUUIDs: [ServiceHierarchy.fitnessMachineServiceUUID], withDuplicates: true)
        }
    }

    func centralManager(_ central: CBCentralManager, didDiscover peripheral: CBPeripheral, advertisementData: [String : Any], rssi RSSI: NSNumber) {
        if self.rssiIsValid(RSSI) {
            updateDevice(Device(withPeripheral: peripheral, RSSI.int32Value)!)
        }
        self.delegate.scannerUpdated()
    }

    private func buildScanOptions(forServices services: [CBUUID]) -> NSDictionary {
        return [CBCentralManagerScanOptionAllowDuplicatesKey: allowDuplicates, CBCentralManagerScanOptionSolicitedServiceUUIDsKey: services]
    }

    fileprivate func rssiIsValid(_ RSSI: NSNumber) -> Bool {
        return RSSI != INVALID_RSSI && RSSI.intValue > self.rssiThreshold
    }
}
