/*
 This software is subject to the license described in the license.txt file included with this software distribution. You may not use this file except in compliance with this license.
 Copyright © Dynastream Innovations Inc. 2017
 All rights reserved.
 */

import UIKit
import CoreBluetooth

class DeviceTableViewController: UIViewController, DeviceScanManagerDelegate, UITableViewDelegate, UITableViewDataSource {

    private let tableCellIdentifier: String = "DeviceTableViewCell"
    private var deviceScanner: DeviceScanManager?

    @IBOutlet var devicesTable: UITableView!

    @IBAction func unwindToDeviceTable(sender: UIStoryboardSegue) {
        if let source = sender.source as? FiltersViewController {
            self.deviceScanner?.setRssiThreshod(Int(source.rssiSlider.value))
        }
        self.deviceScanner?.reopenScan()
    }

    required init?(coder aDecoder: NSCoder) {
        super.init(coder: aDecoder)
    }

    override func viewDidLoad() {
        super.viewDidLoad()
        self.deviceScanner = DeviceScanManager(withDelegate: self)
        setupRefreshController()
    }

    override func viewWillAppear(_ animated: Bool) {
        self.devicesTable.delegate = self
        self.devicesTable.dataSource = self
        self.reloadTable(isHardReload: true)
        self.deviceScanner?.reopenScan()
    }

    override func viewWillDisappear(_ animated: Bool) {
        self.deviceScanner?.closeScan()
        super.viewWillDisappear(animated)
    }

    override func prepare(for segue: UIStoryboardSegue, sender: Any?) {
        if let deviceDataViewController = segue.destination as? DataViewController {
            guard let DeviceCell = sender as? DeviceTableViewCell else {
                fatalError("Unexpected sender: \(String(describing: sender))")
            }
            guard let selectedRowIndex = devicesTable.indexPath(for: DeviceCell)?.row else {
                fatalError("The selected row is not displayed by the table")
            }
            deviceDataViewController.deviceConnectionManager?.setConnectedDevice(with: self.deviceScanner?.getDevice(forIndex: selectedRowIndex).peripheral)
            deviceDataViewController.deviceConnectionManager?.btCentralManager = self.deviceScanner?.btCentralManager
        } else if let filtersViewController = segue.destination as? FiltersViewController {
            filtersViewController.rssiValue = Float((self.deviceScanner?.rssiThreshold)!)
        }
    }

    func scannerUpdated() {
        self.reloadTable()
    }

    func tableView(_ tableView: UITableView, didSelectRowAt indexPath: IndexPath) {
        self.deviceScanner?.closeScan()
        devicesTable.deselectRow(at: indexPath, animated: true)
    }

    func tableView(_ tableView: UITableView, numberOfRowsInSection section: Int) -> Int {
        return self.deviceScanner?.numberOfDevices() ?? 0
    }

    func tableView(_ tableView: UITableView, cellForRowAt indexPath: IndexPath) -> UITableViewCell {
        guard let cell = tableView.dequeueReusableCell(withIdentifier: self.tableCellIdentifier, for: indexPath) as? DeviceTableViewCell  else {
            fatalError("Cell is not an instance of DeviceTableViewCell.")
        }
        cell.nameLabel.text = deviceScanner?.devices[indexPath.row].peripheralName()
        cell.pickRssiIcon((deviceScanner?.devices[indexPath.row].signalStrength)!)
        cell.cellBackground.layer.cornerRadius = cell.frame.height / 4
        cell.cellBackground.layer.masksToBounds = true
        cell.cellBackground.backgroundColor = UserInterfaceElements.cardColor
        return cell
    }

    func reloadTable(isHardReload hardReload: Bool = false) {
        for device in (deviceScanner?.devices)! {
            if device.timedOut() || hardReload {
                let removeIndex = (self.deviceScanner?.devices)!.index(of: device)!
                self.deviceScanner?.devices.remove(at: removeIndex)
            }
        }
        devicesTable.reloadData()
    }

    func refreshTable(_ refreshControl: UIRefreshControl) {
        deviceScanner?.devices.removeAll()
        refreshControl.endRefreshing()
        self.deviceScanner?.openScan(forUUIDs: [ServiceHierarchy.fitnessMachineServiceUUID], withDuplicates: true)
        devicesTable.reloadData()
    }

    private func setupRefreshController() {
        let refreshControl = UIRefreshControl()
        refreshControl.addTarget(self, action: #selector(refreshTable(_:)), for: .valueChanged)
        if #available(iOS 10.0, *) {
            devicesTable.refreshControl = refreshControl
        } else {
            devicesTable.backgroundView = refreshControl
        }
    }
}
