/*
 This software is subject to the license described in the license.txt file included with this software distribution. You may not use this file except in compliance with this license.
 Copyright © Dynastream Innovations Inc. 2017
 All rights reserved.
 */

//
//  FitFileManager.mm
//  g.fit
//

#import "FitFileManager.h"
#import "FitEncode.h"
#import "FitDecode.h"

#include <list>
#include "fit_file_id_mesg.hpp"
#include "fit_file_id_mesg_listener.hpp"
#include "fit_record_mesg_listener.hpp"
#include "fit_mesg_broadcaster.hpp"
#include "fit_mesg.hpp"
#include "fit.hpp"

std::list<fit::RecordMesg> recordMessages;
std::list<fit::SessionMesg> sessionMessages;
FILE *file;
fit::FileIdMesg fileId;
fit::DeveloperDataIdMesg devId;

@interface FitFileManager ()

- (const char *)readOnlyParam;
- (const char *)writeOnlyParam;
+ (NSString *)stringForWString:(FIT_WSTRING)wString;

@property FitEncode *fitEncode;
@property NSString *fileName;
@property FILE *file;
@property FIT_UINT8 *fitStatus;
@property NSTimer *requestTimer;

@end

@implementation FitFileManager

- (id)init: (id<FitFileManagerDelegate>) delegate
{
    self = [super init];
    self.delegate = delegate;
    return self;
}

- (const char *)readOnlyParam { return "rb"; }

- (const char *)writeOnlyParam { return "wb+"; }

- (FILE *)openFileWithParams:(const char *)params withFileName: (NSString*) fileName
{
    NSString *docsPath = [NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES) lastObject];
    NSString *filePath = [docsPath stringByAppendingPathComponent: fileName];
    return fopen([filePath UTF8String], params);
}

+ (NSString *)stringForWString:(FIT_WSTRING)wString
{
    return [[NSString alloc] initWithBytes:wString.data() length:wString.size() * sizeof(wchar_t) encoding:NSUTF32LittleEndianStringEncoding];
}

- (void)startRecording
{
    self.isRecording = true;
    self.requestTimer = [NSTimer scheduledTimerWithTimeInterval:1.0f target:self selector:@selector(requestRecordHook) userInfo:nil repeats:YES];
}

- (void)stopRecording
{
    self.isRecording = false;
    [self.requestTimer invalidate];
    self.requestTimer = nil;
}

- (void) completeSessionWithName: (NSString *) fileName
{
    [self requestSessionHook];
    [self stopRecording];
    [self encodeFile:fileName];
}

- (void)requestRecordHook
{
    [self.delegate fitRequestedRecordMessage];
}

- (void)requestSessionHook
{
    [self.delegate fitRequestedSessionMessages];
}

- (void)encodeFile:(NSString *)fileName
{
    [self encodeFitFile: fileName];
}

- (FIT_UINT8)encodeFitFile: (NSString *) fileName
{
    FILE *file;
    FitEncode *fitEncoder = [[FitEncode alloc] initWithVersion:fit::ProtocolVersion::V10];

    if( ( file = [self openFileWithParams:[self writeOnlyParam] withFileName: [NSString stringWithFormat:@"%@%@", fileName, @".fit"]] ) == NULL)
    {
        return -1;
    }

    fit::FileIdMesg fileId;
    fileId.SetType(FIT_FILE_ACTIVITY);
    fileId.SetManufacturer(FIT_MANUFACTURER_DYNASTREAM);
    fileId.SetProduct(1231);
    fileId.SetSerialNumber(12345);

    fit::DeveloperDataIdMesg devId;
    for (FIT_UINT8 i = 0; i < 16; i++)
    {
        devId.SetApplicationId(i, i);
    }
    devId.SetDeveloperDataIndex(0);

    [fitEncoder Open:file];
    [fitEncoder WriteMesg:fileId];
    [fitEncoder WriteMesg:devId];

    for (auto recordMessage : recordMessages)
    {
        [fitEncoder WriteMesg:recordMessage];
    }
    for (auto sessionMessage : sessionMessages)
    {
        [fitEncoder WriteMesg:sessionMessage];
    }
    if (![fitEncoder Close])
    {
        return -1;
    }
    fclose(file);
    file = NULL;

    return 0;
}

- (void)addSessionMessage: (FitTypeValuePair *) samples numberOfSamples: (int) numSamples
{
    fit::SessionMesg newSessionMessage;
    for (int index = 0; index < numSamples; index++)
    {
        [self setValueInSessionMessage:samples[index] forMessage: &newSessionMessage];
    }
    sessionMessages.push_back(newSessionMessage);
}

- (void)addRecordMessage: (FitTypeValuePair *) samples numberOfSamples: (int) numSamples
{
    fit::RecordMesg newRecordMessage;
    for (int index = 0; index < numSamples; index++)
    {
        [self setValueInRecordMessage: samples[index] forMessage: &newRecordMessage];
    }
    recordMessages.push_back(newRecordMessage);
}

- (void) setValueInRecordMessage: (FitTypeValuePair) typeValuePair forMessage: (fit::RecordMesg *) message
{
    switch (typeValuePair.messageType) {
        case SPEED_RECORD_MESSAGE:
            message->SetSpeed(typeValuePair.value / 3.6);
            break;
        case DISTANCE_RECORD_MESSAGE:
            message->SetDistance(typeValuePair.value);
            break;
        case CALORIES_RECORD_MESSAGE:
            message->SetCalories(typeValuePair.value);
            break;
        case RESISTANCE_RECORD_MESSAGE:
            message->SetResistance(typeValuePair.value);
            break;
        case GRADE_RECORD_MESSAGE:
            message->SetGrade(typeValuePair.value);
            break;
        case CADENCE_RECORD_MESSAGE:
            message->SetCadence(typeValuePair.value);
            break;
        case POWER_RECORD_MESSAGE:
            message->SetPower(typeValuePair.value);
            break;
        case HEART_RATE_RECORD_MESSAGE:
            message->SetHeartRate(typeValuePair.value);
            break;
        case TIMESTAMP_RECORD_MESSAGE:
            message->SetTimestamp(typeValuePair.value);
        default:
            break;
    }
}

- (void) setValueInSessionMessage: (FitTypeValuePair) typeValuePair forMessage: (fit::SessionMesg *) message
{
    switch (typeValuePair.messageType) {
        case DISTANCE_SESSION_MESSAGE:
            message->SetTotalDistance(typeValuePair.value);
            break;
        case CALORIES_SESSION_MESSAGE:
            message->SetTotalCalories(typeValuePair.value);
            break;
        case ASCENT_SESSION_MESSAGE:
            message->SetTotalAscent(typeValuePair.value);
            break;
        case DESCENT_SESSION_MESSAGE:
            message->SetTotalDescent(typeValuePair.value);
            break;

        default:
            break;
    }
}

@end
