/*
 This software is subject to the license described in the license.txt file included with this software distribution. You may not use this file except in compliance with this license.
 Copyright © Dynastream Innovations Inc. 2017
 All rights reserved.
 */

import Foundation
import CoreBluetooth

// Defines a structure for implementing and relating services to their characteristics, and characteristics to the data fields that use them
class ServiceHierarchy {

    // Connect a Service (by its uuid) to its characteristics
    struct DefinedService {
        var uuid: CBUUID
        var characteristics: [DefinedCharacteristic]
    }

    // Connect a Characteristic (by its uuid) to the dataFields it is used in
    struct DefinedCharacteristic {
        var uuid: CBUUID
        var dataFields: [DataField]
    }

    // Desciribe a datafield and what to use to decode the data
    struct DataField{
        let fitRecordMessageType: FIT_MESSAGE_TYPE?
        let fitSessionMessageType: FIT_MESSAGE_TYPE?
        var title: String
        var units: String
        var decoder: (Data) -> (Any)
        var forDisplay: Bool = true

        init(title: String, units: String, decoder: @escaping (Data) -> (Any), forDisplay: Bool = true, fitRecordMessageType: FIT_MESSAGE_TYPE? = nil, fitSessionMessageType: FIT_MESSAGE_TYPE? = nil) {
            self.title = title
            self.units = units
            self.decoder = decoder
            self.forDisplay = forDisplay
            self.fitRecordMessageType = fitRecordMessageType
            self.fitSessionMessageType = fitSessionMessageType
        }
    }

    // Search the hierarchy for data fields related to a characteristic
    class func lookupDatafields(forCharacteristicUUID searchUUID: CBUUID) -> [DataField] {
        for service in ServiceHierarchy.services.allValues as! [DefinedService] {
            for characteristic in service.characteristics {
                if characteristic.uuid == searchUUID {
                    return characteristic.dataFields
                }
            }
        }
        return []
    }


    // MARK: - Services Dictionary representing implemented BLE services
    static let services: NSDictionary = [
        fitnessMachineServiceUUID: fitnessMachineService,
        hrServiceUUID: heartRateService,
        batteryServiceUUID: batteryStatusService,
        deviceInformationServiceUUID: deviceInformationService
    ]


    // MARK: - DefinedServices
    static let fitnessMachineService = DefinedService(uuid: fitnessMachineServiceUUID, characteristics: [ftmsFeaturesCharacteristic,

                                                                                                         treadmillDataCharacteristic,
                                                                                                         stationaryBikeDataCharacteristic,
                                                                                                         rowerDataCharacteristic,
                                                                                                         crossTrainerDataCharacteristic,
                                                                                                         stepClimberDataCharacteristic])

    static let heartRateService = DefinedService(uuid: hrServiceUUID, characteristics: [heartRateCharacteristic, heartRateLocationCharacteristic])

    static let batteryStatusService = DefinedService(uuid: batteryServiceUUID, characteristics: [batteryStatusCharacteristic])

    static let deviceInformationService = DefinedService(uuid: deviceInformationServiceUUID, characteristics: [manufacturerNameStringCharacteristic,
                                                                                                         modelNumberStringCharacteristic,
                                                                                                         serialNumberStringCharacteristic,
                                                                                                         hardwareRevisionStringCharacteristic,
                                                                                                         firmwareRevisionStringCharacteristic,
                                                                                                         softwareRevisionStringCharacteristic])


    // MARK: - DefinedCharacteristicsc
    // Fitness Machine
    static let ftmsFeaturesCharacteristic = DefinedCharacteristic(uuid: ftmsFeaturesUUID, dataFields: [ftmsFeaturesDataField])

    static let treadmillDataCharacteristic = DefinedCharacteristic(uuid: treadmillDataUUID, dataFields: [treadmillElapsedTimeDataField,
                                                                                                         treadmillInstantaneousPaceDataField,
                                                                                                         treadmillPositiveElevationGainDataField,
                                                                                                         treadmillNegativeElevationGainDataField,
                                                                                                         treadmillInclinationDataField,
                                                                                                         treadmillTotalDistanceDataField,
                                                                                                         treadmillInstantaneousSpeedDataField,
                                                                                                         treadmillMetabolicEquivalentDataField,
                                                                                                         treadmillTotalEnergyDataField,
                                                                                                         treadmillEnergyPerHourDataField])

    static let stationaryBikeDataCharacteristic = DefinedCharacteristic(uuid: indoorBikeDataUUID, dataFields: [stationaryBikeElapsedTimeDataField,
                                                                                                         stationaryBikeResistanceLevelDataField,
                                                                                                         stationaryBikeInstantaneousCadenceDataField,
                                                                                                         stationaryBikeInstantaneousPowerDataField,
                                                                                                         stationaryBikeInstantaneousSpeedDataField,
                                                                                                         stationaryBikeTotalDistanceDataField,
                                                                                                         stationaryBikeMetabolicEquivalentDataField,
                                                                                                         stationaryBikeTotalEnergyDataField,
                                                                                                         stationaryBikeEnergyPerHourDataField])


    static let rowerDataCharacteristic = DefinedCharacteristic(uuid: rowerDataUUID, dataFields: [rowerElapsedTimeDataField,
                                                                                                         rowerResistanceLevelDataField,
                                                                                                         rowerStrokeCountDataField,
                                                                                                         rowerStrokeRateDataField,
                                                                                                         rowerInstantaneousPowerDataField,
                                                                                                         rowerMetabolicEquivalentDataField,
                                                                                                         rowerTotalEnergyDataField,
                                                                                                         rowerEnergyPerHourDataField,
                                                                                                         rowerTotalDistanceDataField])

    static let crossTrainerDataCharacteristic = DefinedCharacteristic(uuid: crossTrainerDataUUID, dataFields: [crossTrainerElapsedTimeDataField,
                                                                                                         crossTrainerResistanceLevelDataField,
                                                                                                         crossTrainerInclinationDataField,
                                                                                                         crossTrainerPositiveElevationGainDataField,
                                                                                                         crossTrainerStrideCountDataField,
                                                                                                         crossTrainerCadenceDataField,
                                                                                                         crossTrainerInstantaneousPowerDataField,
                                                                                                         crossTrainerMetabolicEquivalentDataField,
                                                                                                         crossTrainerTotalEnergyDataField,
                                                                                                         crossTrainerEnergyPerHour,
                                                                                                         crossTrainerTotalDistanceDataField,
                                                                                                         crossTrainerInstantaneousSpeedDataField])

    static let stepClimberDataCharacteristic = DefinedCharacteristic(uuid: stepClimberDataUUID, dataFields: [stepClimberElapsedTimeDataField,
                                                                                                         stepClimberFloorsDataField,
                                                                                                         stepClimberStepsClimbedCountDataField,
                                                                                                         stepClimberCadenceDataField,
                                                                                                         stepClimberPositiveElevationGainDataField,
                                                                                                         stepClimberMetabolicEquivalentDataField,
                                                                                                         stepClimberTotalEnergyDataField,
                                                                                                         stepClimberEnergyPerHourDataField])
    // Heart Rate
    static let heartRateCharacteristic = DefinedCharacteristic(uuid: hrCharacteristicUUID, dataFields: [heartRateDataField])
    static let heartRateLocationCharacteristic = DefinedCharacteristic(uuid: hrSensorLocationCharacteristicUUID, dataFields: [heartRateLocationDataField])
    // Battery
    static let batteryStatusCharacteristic = DefinedCharacteristic(uuid: batteryLevelCharacteristicUUID, dataFields: [batteryStatusDataField])
    // Device Information
    static let manufacturerNameStringCharacteristic = DefinedCharacteristic(uuid: manufacturerNameStringUUID, dataFields: [manufacturerNameDataField])
    static let modelNumberStringCharacteristic = DefinedCharacteristic(uuid: modelNumberStringUUID, dataFields: [modelNumberDataField])
    static let serialNumberStringCharacteristic = DefinedCharacteristic(uuid: serialNumberStringUUID, dataFields: [])
    static let hardwareRevisionStringCharacteristic = DefinedCharacteristic(uuid: hardwareRevisionStringUUID, dataFields: [])
    static let firmwareRevisionStringCharacteristic = DefinedCharacteristic(uuid: firmwareRevisionStringUUID, dataFields: [firmwareRevivisionDataField])
    static let softwareRevisionStringCharacteristic = DefinedCharacteristic(uuid: softwareRevisionStringUUID, dataFields: [softwareRevisionDataField])


    // MARK: - Datafields
    // Fitness Machine Data Fields
    // Treadmill
    static let treadmillInstantaneousPaceDataField          = DataField(title: "Instantaneous Pace", units: "m", decoder: Decoders.decodeTreadmillInstantaneousPace, forDisplay: true)
    static let treadmillPositiveElevationGainDataField      = DataField(title: "Positive Elev. Gain", units: "m", decoder: Decoders.decodeTreadmillPositiveElevationGain, forDisplay: true, fitSessionMessageType: ASCENT_SESSION_MESSAGE)
    static let treadmillNegativeElevationGainDataField      = DataField(title: "Negative Elev. Gain", units: "m", decoder: Decoders.decodeTreadmillNegativeElevationGain, forDisplay: true, fitSessionMessageType: DESCENT_RECORD_MESSAGE)
    static let treadmillInclinationDataField                = DataField(title: "Inclination", units: "%", decoder: Decoders.decodeTreadmillInclination, forDisplay: true, fitRecordMessageType: GRADE_RECORD_MESSAGE)
    static let treadmillInstantaneousSpeedDataField         = DataField(title: "Instantaneous Speed", units: "km/h", decoder: Decoders.decodeTreadmillInstantaneousSpeed, forDisplay: true, fitRecordMessageType: SPEED_RECORD_MESSAGE)
    static let treadmillTotalDistanceDataField              = DataField(title: "Total Distance", units: "m", decoder: Decoders.decodeTreadmillTotalDistance, forDisplay: true, fitRecordMessageType: DISTANCE_RECORD_MESSAGE, fitSessionMessageType: DISTANCE_SESSION_MESSAGE)
    static let treadmillMetabolicEquivalentDataField        = DataField(title: "Metabolic Equivalent", units: "MET", decoder: Decoders.decodeTreadmillMetabolicEquivalent, forDisplay: true)
    static let treadmillTotalEnergyDataField                = DataField(title: "Total Energy", units: "kCal", decoder: Decoders.decodeTreadmillTotalEnergy, forDisplay: true, fitRecordMessageType: CALORIES_RECORD_MESSAGE, fitSessionMessageType: CALORIES_SESSION_MESSAGE)
    static let treadmillEnergyPerHourDataField              = DataField(title: "Energy/Hour", units: "kCal", decoder: Decoders.decodeTreadmillEnergyPerHour, forDisplay: true)
    static let treadmillElapsedTimeDataField                   = DataField(title: "Elapsed Time", units: "s", decoder: Decoders.decodeTreadmillElapsedTime, forDisplay: true, fitRecordMessageType: TIMESTAMP_RECORD_MESSAGE)

    // Stationary Bike
    static let stationaryBikeResistanceLevelDataField       = DataField(title: "Resistance Level", units: "", decoder: Decoders.decodeStationaryBikeResistanceLevel, forDisplay: true, fitRecordMessageType: RESISTANCE_RECORD_MESSAGE)
    static let stationaryBikeInstantaneousCadenceDataField  = DataField(title: "Instantaneous Cadence", units: "rpm", decoder: Decoders.decodeStationaryBikeInstantaneousCadence, forDisplay: true, fitRecordMessageType: CADENCE_RECORD_MESSAGE)
    static let stationaryBikeInstantaneousPowerDataField    = DataField(title: "Instantaneous Power", units: "W", decoder: Decoders.decodeStationaryBikeInstantaneousPower, forDisplay: true, fitRecordMessageType: POWER_RECORD_MESSAGE)
    static let stationaryBikeTotalDistanceDataField         = DataField(title: "Total Distance", units: "m", decoder: Decoders.decodeStationaryBikeTotalDistance, forDisplay: true, fitRecordMessageType: DISTANCE_RECORD_MESSAGE, fitSessionMessageType: DISTANCE_SESSION_MESSAGE)
    static let stationaryBikeInstantaneousSpeedDataField    = DataField(title: "Instantaneous Speed", units: "km/h", decoder: Decoders.decodeStationaryBikeInstantaneousSpeed, forDisplay: true, fitRecordMessageType: SPEED_RECORD_MESSAGE)
    static let stationaryBikeMetabolicEquivalentDataField   = DataField(title: "Metabolic Equivalent", units: "MET", decoder: Decoders.decodeStationaryBikeMetabolicEquivalent, forDisplay: true)
    static let stationaryBikeTotalEnergyDataField           = DataField(title: "Total Energy", units: "kCal", decoder: Decoders.decodeStationaryBikeTotalEnergy, forDisplay: true, fitRecordMessageType: CALORIES_RECORD_MESSAGE, fitSessionMessageType: CALORIES_SESSION_MESSAGE)
    static let stationaryBikeEnergyPerHourDataField         = DataField(title: "Energy/Hour", units: "kCal", decoder: Decoders.decodeStationaryBikeEnergyPerHour, forDisplay: true)
    static let stationaryBikeElapsedTimeDataField                   = DataField(title: "Elapsed Time", units: "s", decoder: Decoders.decodeStationaryBikeElapsedTime, forDisplay: true, fitRecordMessageType: TIMESTAMP_RECORD_MESSAGE)

    // Rower
    static let rowerResistanceLevelDataField                = DataField(title: "Resistance Level", units: "", decoder: Decoders.decodeRowerResistanceLevel, forDisplay: true, fitRecordMessageType: RESISTANCE_RECORD_MESSAGE)
    static let rowerStrokeCountDataField                    = DataField(title: "Stroke Count", units: "", decoder: Decoders.decodeRowerStrokeCount, forDisplay: true)
    static let rowerStrokeRateDataField                     = DataField(title: "Stroke Rate", units: "/min", decoder: Decoders.decodeRowerStrokeRate, forDisplay: true)
    static let rowerInstantaneousPowerDataField             = DataField(title: "Instantaneous Power", units: "W", decoder: Decoders.decodeRowerInstantaneousPower, forDisplay: true, fitRecordMessageType: POWER_RECORD_MESSAGE)
    static let rowerMetabolicEquivalentDataField            = DataField(title: "Metabolic Equivalent", units: "MET", decoder: Decoders.decodeRowerMetabolicEquivalent, forDisplay: true)
    static let rowerTotalEnergyDataField                    = DataField(title: "Total Energy", units: "kCal", decoder: Decoders.decodeRowerTotalEnergy, forDisplay: true, fitRecordMessageType: CALORIES_RECORD_MESSAGE, fitSessionMessageType: CALORIES_SESSION_MESSAGE)
    static let rowerEnergyPerHourDataField                  = DataField(title: "Energy/Hour", units: "kCal", decoder: Decoders.decodeRowerEnergyPerHour, forDisplay: true)
    static let rowerElapsedTimeDataField                    = DataField(title: "Elapsed Time", units: "s", decoder: Decoders.decodeRowerElapsedTime, forDisplay: true, fitRecordMessageType: TIMESTAMP_RECORD_MESSAGE)
    static let rowerTotalDistanceDataField                  = DataField(title: "Total Distance", units: "m", decoder: Decoders.decodeRowerTotalDistance, forDisplay: true, fitRecordMessageType: DISTANCE_RECORD_MESSAGE, fitSessionMessageType: DISTANCE_SESSION_MESSAGE)

    // Elliptical
    static let crossTrainerInstantaneousSpeedDataField      = DataField(title: "Instataneous Speed", units: "km/h", decoder: Decoders.decodeCrossTrainerInstantaneousSpeed, forDisplay: true, fitRecordMessageType: SPEED_RECORD_MESSAGE)
    static let crossTrainerResistanceLevelDataField         = DataField(title: "Resistance Level", units: "", decoder: Decoders.decodeCrossTrainerResistanceLevel, forDisplay: true, fitRecordMessageType: RESISTANCE_RECORD_MESSAGE)
    static let crossTrainerInclinationDataField             = DataField(title: "Inclination", units: "%", decoder: Decoders.decodeCrossTrainerInclination, forDisplay: true)
    static let crossTrainerPositiveElevationGainDataField   = DataField(title: "Elevation Gain", units: "m", decoder: Decoders.decodeCrossTrainerPositiveElevationGain, forDisplay: true, fitSessionMessageType: ASCENT_SESSION_MESSAGE)
    static let crossTrainerStrideCountDataField             = DataField(title: "Stride Count", units: "", decoder: Decoders.decodeCrossTrainerStrideCount, forDisplay: true)
    static let crossTrainerCadenceDataField                 = DataField(title: "Cadence", units: "step/min", decoder: Decoders.decodeCrossTrainerCadence, forDisplay: true, fitRecordMessageType: CADENCE_RECORD_MESSAGE)
    static let crossTrainerInstantaneousPowerDataField      = DataField(title: "Instantaneous Power", units: "W", decoder: Decoders.decodeCrossTrainerInstantaneousPower, forDisplay: true, fitRecordMessageType: POWER_RECORD_MESSAGE)
    static let crossTrainerMetabolicEquivalentDataField     = DataField(title: "Metabolic Equivalent", units: "MET", decoder: Decoders.decodeCrossTrainerMetabolicEquivalent, forDisplay: true)
    static let crossTrainerTotalEnergyDataField             = DataField(title: "Total Energy", units: "kCal", decoder: Decoders.decodeCrossTrainerTotalEnergy, forDisplay: true, fitRecordMessageType: CALORIES_RECORD_MESSAGE, fitSessionMessageType: CALORIES_SESSION_MESSAGE)
    static let crossTrainerEnergyPerHour                    = DataField(title: "Energy/Hour", units: "kCal", decoder: Decoders.decodeCrossTrainerEnergyPerHour, forDisplay: true)
    static let crossTrainerTotalDistanceDataField           = DataField(title: "Total Distance", units: "m", decoder: Decoders.decodeCrossTrainerTotalDistance, forDisplay: true, fitRecordMessageType: DISTANCE_RECORD_MESSAGE, fitSessionMessageType: DISTANCE_SESSION_MESSAGE)
    static let crossTrainerElapsedTimeDataField                   = DataField(title: "Elapsed Time", units: "s", decoder: Decoders.decodeCrossTrainerElapsedTime, forDisplay: true, fitRecordMessageType: TIMESTAMP_RECORD_MESSAGE)

    // Step Climber
    static let stepClimberFloorsDataField                   = DataField(title: "Floors Climbed", units: "", decoder: Decoders.decodeStepClimberFloorsClimbed, forDisplay: true)
    static let stepClimberStepsClimbedCountDataField        = DataField(title: "Steps Climbed", units: "", decoder: Decoders.decodeStepClimberStepsClimbedCount, forDisplay: true)
    static let stepClimberCadenceDataField                  = DataField(title: "Cadence", units: "steps/min", decoder: Decoders.decodeStepClimberCadence, forDisplay: true, fitRecordMessageType: CADENCE_RECORD_MESSAGE)
    static let stepClimberPositiveElevationGainDataField    = DataField(title: "Positive Elevation Gain", units: "m", decoder: Decoders.decodeStepClimberPositiveElevationGain, forDisplay: true, fitSessionMessageType: ASCENT_SESSION_MESSAGE)
    static let stepClimberMetabolicEquivalentDataField      = DataField(title: "Metabolic Equivalent", units: "MET", decoder: Decoders.decodeStepClimberMetabolicEquivalent, forDisplay: true)
    static let stepClimberTotalEnergyDataField              = DataField(title: "Total Energy", units: "kCal", decoder: Decoders.decodeStepClimberTotalEnergy, forDisplay: true, fitRecordMessageType: CALORIES_RECORD_MESSAGE, fitSessionMessageType: CALORIES_SESSION_MESSAGE)
    static let stepClimberEnergyPerHourDataField            = DataField(title: "Energy/Hour", units: "kCal", decoder: Decoders.decodeStepClimberEnergyPerHour, forDisplay: true)
    static let stepClimberElapsedTimeDataField              = DataField(title: "Elapsed Time", units: "s", decoder: Decoders.decodeStepClimberElapsedTime, forDisplay: true, fitRecordMessageType: TIMESTAMP_RECORD_MESSAGE)

    // Fitness Machine Features
    static let ftmsFeaturesDataField                        = DataField(title: "Features", units: "", decoder: Decoders.decodeFtmsFeatures, forDisplay: false)

    // Heart Rate
    static let heartRateDataField                           = DataField(title: "Heart Rate", units: "bpm", decoder: Decoders.decodeHRValue, forDisplay: true, fitRecordMessageType: HEART_RATE_RECORD_MESSAGE)
    static let heartRateLocationDataField                   = DataField(title: "HR Sensor Location", units: "", decoder: Decoders.decodeHRLocation, forDisplay: true)

    // Battery
    static let batteryStatusDataField                       = DataField(title: "Battery", units: "%", decoder: Decoders.decodeBatteryStatus, forDisplay: true)

    // Device Information
    static let manufacturerNameDataField                    = DataField(title: "Manufacturer Name", units: "", decoder: Decoders.decodeUTF8s, forDisplay: false)
    static let modelNumberDataField                         = DataField(title: "Model Number", units: "", decoder: Decoders.decodeUTF8s, forDisplay: false)
    static let firmwareRevivisionDataField                  = DataField(title: "Firmware Revision", units: "", decoder: Decoders.decodeUTF8s, forDisplay: false)
    static let softwareRevisionDataField                    = DataField(title: "Software Revision", units: "", decoder: Decoders.decodeUTF8s, forDisplay: false)


    // MARK: - Identifiers
    // Fitness Machine
    static let fitnessMachineServiceUUID = CBUUID(string: "1826")
    static let ftmsFeaturesUUID = CBUUID(string: "2ACC")
    static let treadmillDataUUID = CBUUID(string: "2ACD")
    static let crossTrainerDataUUID = CBUUID(string: "2ACE")
    static let stepClimberDataUUID = CBUUID(string: "2ACF")
    static let stairClimberDataUUID = CBUUID(string: "2AD0")
    static let rowerDataUUID = CBUUID(string: "2AD1")
    static let indoorBikeDataUUID = CBUUID(string: "2AD2")
    static let trainingStatusUUID = CBUUID(string: "2AD3")

    // HRS Identifiers
    static let hrServiceUUID = CBUUID(string: "180D")
    static let hrCharacteristicUUID = CBUUID(string: "2A37")
    static let hrSensorLocationCharacteristicUUID = CBUUID(string: "2A38")


    // Battery Identifiers
    static let batteryServiceUUID = CBUUID(string: "180F")
    static let batteryLevelCharacteristicUUID = CBUUID(string: "2A19")

    // Device Information Identifiers
    static let deviceInformationServiceUUID = CBUUID(string: "180A")
    static let manufacturerNameStringUUID = CBUUID(string: "2A29")
    static let modelNumberStringUUID = CBUUID(string: "2A24")
    static let serialNumberStringUUID = CBUUID(string: "2A25")
    static let hardwareRevisionStringUUID = CBUUID(string: "2A27")
    static let firmwareRevisionStringUUID = CBUUID(string: "2A26")
    static let softwareRevisionStringUUID = CBUUID(string: "2A28")
}
